#!/usr/bin/python3

import argparse
import os
import shutil
import random
import re

def confirm(prompt=None, default=False):
    prompt = prompt or 'Confirm'
    prompt = f'{prompt} (y/n): '
    while True:
        response = input(prompt).lower()
        if not response:
            return default
        if response in {'y', 'yes'}:
            return True
        if response in {'n', 'no'}:
            return False
        print('Invalid input. Please enter "y" or "n".')


def generate_normalized_filename(src_path, source_folder):
    relative_path = os.path.relpath(src_path, start=source_folder)
    base_name = os.path.splitext(relative_path)[0]
    base_name = re.sub(r'[^a-zA-Z0-9]+', '_', base_name).lower()
    return f"{base_name}.mp3"


def copy_mp3_files(source_folder, destination_folder, order_mode, match_string):
    mp3_files = []
    for root, _, files in os.walk(source_folder):
        for filename in files:
            if filename.lower().endswith('.mp3'):
                src_path = os.path.join(root, filename)
                relative_path = os.path.relpath(src_path, start=source_folder)
                if not match_string or match_string.lower() in relative_path.lower():
                    dest_filename = generate_normalized_filename(src_path, source_folder)
                    mp3_files.append((src_path, dest_filename))

    if order_mode == "track":
        random.shuffle(mp3_files)
    elif order_mode == "artist":
        artist_names = set(os.path.dirname(os.path.relpath(src, start=source_folder)).replace(os.sep, '_') for src, _ in mp3_files)
        for artist_name in artist_names:
            files_in_artist_folder = [(src, dest) for src, dest in mp3_files if artist_name == os.path.dirname(os.path.relpath(src, start=source_folder)).replace(os.sep, '_')]
            random.shuffle(files_in_artist_folder)
            mp3_files = [f for f in mp3_files if f not in files_in_artist_folder] + files_in_artist_folder

    for src, dest in mp3_files:
        dest_path = os.path.join(destination_folder, dest)
        shutil.copy(src, dest_path)
        print(f"Copying '{src}' to '{dest}'")

def clear_destination_folder(destination_folder):
    dest_files = os.listdir(destination_folder)
    if dest_files:
        print(f"Destination folder '{destination_folder}' is not empty.")
        if not confirm('Do you want to proceed and clear the destination folder?', default=False):
            exit()

        # Clear the destination folder files only
        for file in dest_files:
            file_path = os.path.join(destination_folder, file)
            if os.path.isfile(file_path):
                os.remove(file_path)
            elif os.path.isdir(file_path):
                shutil.rmtree(file_path)

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Copy MP3 files from a source folder to a destination folder.")
    parser.add_argument("--src", help="Source folder path where MP3 files will be searched.")
    parser.add_argument("--dest", help="Destination folder path where MP3 files will be copied.")
    parser.add_argument("--order", choices=["track", "artist"], default="track", help="Order mode for copying the MP3 files.")
    parser.add_argument("--match", help="Matching string to filter files based on relative path (case insensitive).", default="")
    args = parser.parse_args()

    source_folder = args.src or "/Users/toki/Music/MP3"
    destination_folder = args.dest or "/Volumes/MP3"
    order_mode = args.order
    match_string = args.match

    if not os.path.exists(destination_folder):
        os.makedirs(destination_folder)
    else:
        clear_destination_folder(destination_folder)

    copy_mp3_files(source_folder, destination_folder, order_mode, match_string)
