@tool
extends MeshInstance3D

@export var height_map: Texture2D:
	set(value):
		height_map = value
		draw()
		
@export var max_height: float = 1.0:
	set(value):
		max_height = value
		draw()
		
@export var plane_size: float = 1024.0:
	set(value):
		plane_size = value;
		draw();
		
@export_range(2, 512) var resolution: int = 128:
	set(value):
		resolution = value;
		draw();
		
@export var material: Material:
	set(value):
		material = value;
		draw();

@export var redraw: bool = false:
	set(value):
		draw();
		redraw = false;

func draw():
	if not Engine.is_editor_hint():
		return
		
	if not height_map:
		return

	var st = SurfaceTool.new()
	st.begin(Mesh.PRIMITIVE_TRIANGLES)

	var image = height_map.get_image()
	var image_size = image.get_size()

	var image_sx = float(image_size.x) / resolution
	var image_sy = float(image_size.y) / resolution
	var plane_scale = plane_size  / resolution
	var offset = plane_size * -0.5

	for x in range(resolution):
		for z in range(resolution):
			var ix = float(x) * image_sx
			var iy = float(z) * image_sy
			var p = image.get_pixel(ix, iy)
			var h = p.b * max_height
			st.set_uv(Vector2(x / float(resolution), z / float(resolution)))
			st.add_vertex(Vector3(x * plane_scale + offset, h, z * plane_scale + offset))
	
	for x in range(resolution - 1):
		for z in range(resolution - 1):
			var vi = x * resolution + z
			st.add_index(vi)
			st.add_index(vi + resolution)
			st.add_index(vi + 1)
			st.add_index(vi + 1)
			st.add_index(vi + resolution)
			st.add_index(vi + resolution + 1)

	st.generate_normals()
	var st_mesh = st.commit()
	st_mesh.resource_name = "Mesh"
	st_mesh.surface_set_material(0, material)
	set_mesh(st_mesh)
	_clear_collisions()
	call_deferred("_bake_collisions")
	
	
func _clear_collisions():
	for i in range(get_child_count()):
		if get_child(i) is StaticBody3D:
			get_child(i).queue_free()
	
	
func _bake_collisions():
	_clear_collisions()
	var collider = CollisionShape3D.new()
	var static_body = StaticBody3D.new()
	static_body.add_child(collider)
	add_child(static_body)
	static_body.owner = get_tree().edited_scene_root
	collider.owner = get_tree().edited_scene_root
	static_body.name = "Collision"
	collider.name = "Collider"
	collider.shape = mesh.create_trimesh_shape()
