import SwiftUI

enum DreadFrame : Int { case
    HOME,
    CHANNEL,
    POST,
    MESSAGES,
    PROFILE,
    INVALID,
    MAX
}

struct DreadRoute : Hashable {
    let url: URL
    let frame: DreadFrame
    var name: String?
    var icon: String?
    
    func hash(into hasher: inout Hasher) {
        hasher.combine(toString())
    }
    
    func toString() -> String {
        return url.absoluteString
    }
}

struct DreadLink : Hashable {
    
    var name: String
    var icon: String
    var frame: DreadFrame
    var link: String = "/"
    var enabled: Bool = true
    var view: (any View)? = nil
    
    func hash(into hasher: inout Hasher) {
        hasher.combine(frame)
    }
    
    static func == (lhs: DreadLink, rhs: DreadLink) -> Bool {
        return Routes.areEqual(lhs.link, rhs.link)
    }
}

class Links {
    static let LINK_CHANNELS = DreadLink(
        name: "Channels",
        icon: "list.bullet",
        frame: DreadFrame.CHANNEL,
        link: "/subreddits/"
    )
    static let LINK_CHANNELS_MINE = DreadLink(
        name: "Channels",
        icon: "list.bullet",
        frame: DreadFrame.CHANNEL,
        link: "/subreddits/mine"
    )
    static let LINK_CREATE_POST = DreadLink(
        name: "Post",
        icon: "plus.circle.fill",
        frame: DreadFrame.POST,
        enabled: false
    )
    static let LINK_HOME = DreadLink(
        name: "Home",
        icon: "house.fill",
        frame: DreadFrame.HOME,
        link: "/"
    )
    static let LINK_INBOX = DreadLink(
        name: "Messages",
        icon: "envelope.fill",
        frame: DreadFrame.MESSAGES,
        link: "/message/inbox"
    )
    static let LINK_LOGIN = DreadLink(
        name: "Profile",
        icon: "person.fill",
        frame: DreadFrame.PROFILE,
        link: "/login"
    )
    static let LINK_PROFILE = DreadLink(
        name: "Profile",
        icon: "person.fill",
        frame: DreadFrame.PROFILE,
        link: "/user/$USERNAME"
    )
}

class Routes {
    
    static let BASE_URL = "https://old.reddit.com/"
    static let HOME_URL = BASE_URL
    static let HOME_ROUTE = parseRoute(HOME_URL)
    
    /*
    static let DEFAULT_ROUTES: [DreadRoute] = [
        DreadRoute(link: "/", name: "Home", icon: "house.fill", frame: DreadFrame.HOME),
        DreadRoute(link: "/r/", name: "Channels", icon: "list.bullet", frame: DreadFrame.CHANNEL)
        DreadRoute(link: "/submit", name: "Post", icon: "plus.circle.fill", frame: DreadFrame.POST)
        DreadRoute(link: "/message/inbox", name: "Inbox", icon: "envelope.fill", frame: DreadFrame.MESSAGES)
        DreadRoute(link: "/login", name: "Login", icon: "person.fill")
    ]
    
    static func getRoutes(username: String?) -> [DreadRoute] {
        let result = DEFAULT_ROUTES
        return result
    }
     */
    
    static func areEqual(_ a: String, _ b: String) -> Bool {
        return parseUrl(a).absoluteString == parseUrl(b).absoluteString
    }
    
    static func parseUrl(_ urlStr: String?) -> URL {
        var s = HOME_URL
        if (urlStr != nil) {
            s = urlStr!
            if (s.starts(with: "/")) {
                s = BASE_URL + s.suffix(from: s.index(s.startIndex, offsetBy: 1))
            }
        }
        return URL(string: s)!
    }
    
    static func parseFrame(_ urlStr: String?) -> DreadFrame {
        var result = urlStr == HOME_URL ? DreadFrame.HOME : DreadFrame.INVALID
        let url = parseUrl(urlStr)
        if (url.pathComponents.count > 1) {
            result = urlStr == HOME_URL ? DreadFrame.HOME : DreadFrame.CHANNEL
            let subroute = url.pathComponents[1].lowercased()
            if (subroute == "r") {
                if (url.pathComponents.count > 2) {
                    if (url.pathComponents.count > 3 && url.pathComponents[3] == "comments") {
                        result = DreadFrame.POST
                    } else {
                        result = DreadFrame.CHANNEL
                    }
                } else {
                    result = DreadFrame.INVALID
                }
            }
            else if (subroute == "user" || subroute == "u") {
                if (url.pathComponents.count > 2) {
                    let routeUser = url.pathComponents[2]
                    if (routeUser != DreadService.instance().session.username) {
                        result = DreadFrame.POST
                    } else {
                        result = DreadFrame.PROFILE
                    }
                } else {
                    result = DreadFrame.INVALID
                }
            }
            else if (subroute == "login") {
                result = DreadFrame.PROFILE
            }
            else if (subroute == "message") {
                result = DreadFrame.MESSAGES
            }
        }
        return result
    }
    
    static func parseRoute(_ urlStr: String?) -> DreadRoute {
        let result = DreadRoute(
            url: parseUrl(urlStr),
            frame: parseFrame(urlStr)
        )
        return result
    }
    
    static let DISALLOWED_DOMAINS = [
        "ads.reddit.com",
        "alb.reddit.com",
        "events.reddit.com",
        "events.redditmedia.com",
        "pixel.reddit.com",
        "pixel.redditmedia.com",
        "stats.redditmedia.com",
        "tracker.redditmedia.com",
        "twitter.com",
        "google.com"
    ]
    
    static let DISALLOWED_URLS = [
        "redditmedia.com/gtm"
    ]
    
    static let INTERNAL_DOMAINS = [
        "reddit.com",
        "redditmedia.com",
        "daleblackwood.com"
    ]
    
    static func isStringInList(_ s: String?, _ list: [String]) -> Bool {
        if (s != nil) {
            for item in list {
                if (s!.contains(item)) {
                    return true;
                }
            }
        }
        return false
    }
    
    static func isInternal(url: URL?) -> Bool {
        return url != nil && isDisallowed(url: url) == false && isStringInList(url!.host, INTERNAL_DOMAINS);
    }
    
    static func isDisallowed(url: URL?) -> Bool {
        return url == nil || isStringInList(url!.host, DISALLOWED_DOMAINS) || isStringInList(url!.absoluteString, DISALLOWED_URLS);
    }
    
}
