import UIKit
import WebKit

class DreadView: UIViewController, DreadRouteListener {
    
    private let frame: DreadFrame
    private  let scriptFile = TextFile(forResource: "dreadit", ofType: "js")
    private var webview: WKWebView? = nil
    private var isLoading = true
    var hasLoaded = false
    var currentUrl: URL? = nil
    private var config: WKWebViewConfiguration?
    private var navDelegate: WebNavDelegate?
    private var contentCtrl: WebContentController?
    private var timerShow: Timer?
    
    init(_ frame: DreadFrame) {
        self.frame = frame
        super.init(nibName: nil, bundle: nil)
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        webview = makeWebView()
        view = webview
        navigate(URL(string: "about:blank")!)
        view.backgroundColor = self.traitCollection.userInterfaceStyle == .dark ?
            UIColor.black :
            UIColor.white
        DreadService.instance().routeListeners.add(self)
        if let route = DreadService.instance().getFrameRoute(frame) {
            onRouteChange(route)
        }
    }
    
    func navigate(_ url: URL) {
        if (url != currentUrl) {
            setLoading(true)
            currentUrl = url
            webview!.load(URLRequest(url: url))
        }
    }

    func onRouteChange(_ route: DreadRoute) {
        if (route.frame == frame) {
            navigate(route.url)
        }
    }
    
    func onLoadBegin(_ url: URL) -> Bool {
        if (Routes.isInternal(url: url)) {
            let route = Routes.parseRoute(url.absoluteString)
            if (route.frame == DreadFrame.INVALID || route.frame == frame) {
                setLoading(true)
                return true
            } else {
                DreadService.instance().navigate(route)
                return false
            }
        } else {
            UIApplication.shared.open(url)
            return false
        }
    }
    
    func onLoaded() {
        let jsScript = scriptFile.get()
        if (jsScript != nil) {
            let parent = self
            webview!.evaluateJavaScript(jsScript!) { (result, error) in
                if let error = error {
                    print("Error executing JavaScript: \(error)")
                    return
                }
                parent.webview!.evaluateJavaScript("dreadit.getUser()") { (result, error) in
                    if let error = error {
                        print("Error executing JavaScript: \(error)")
                        return
                    }
                    if let jsonString = result as? String {
                        DreadService.instance().setSession(jsonString)
                    }
                    parent.setLoading(false)
                }
            }
        }
    }
    
    func onMessage(_ message: [String : AnyObject]) {
        for entry in message {
            if (entry.key == "navigate") {
                let route = Routes.parseRoute(entry.value as? String)
                DreadService.instance().navigate(route)
            }
        }
    }
    
    func setLoading(_ value: Bool) {
        isLoading = value
        let webview = self.webview
        if (value == false) {
            webview!.isHidden = value
        }
        Timer.scheduledTimer(withTimeInterval: 0.1, repeats: false) { (timer) in
            webview?.isHidden = value
        }
    }
    
    func makeWebView() -> WKWebView {
        let config = WKWebViewConfiguration()
        let contentCtrl = WebContentController(self)
        config.userContentController = contentCtrl
        let webview = WKWebView(frame: .zero, configuration: config)
        navDelegate = WebNavDelegate(self)
        webview.navigationDelegate = navDelegate
        webview.scrollView.isScrollEnabled = true
        webview.scrollView.isPagingEnabled = false
        webview.scrollView.minimumZoomScale = 1.0
        webview.scrollView.maximumZoomScale = 1.0
        webview.pageZoom = 1.0
        webview.customUserAgent = "Mozilla/5.0 (Linux; Android 8.0.0; SM-G955U Build/R16NW) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/87.0.4280.141 Mobile Safari/537.36"
        return webview
    }
    
    class WebNavDelegate : NSObject, WKNavigationDelegate {
        
        var parent: DreadView
        
        init(_ parent: DreadView) {
            self.parent = parent
        }
            
        func webView(_ webview: WKWebView, decidePolicyFor navigationAction: WKNavigationAction, decisionHandler: @escaping (WKNavigationActionPolicy) -> Void) {
            let url = navigationAction.request.url
            if (url == nil) {
                decisionHandler(.cancel)
                return
            }
            if (Routes.isDisallowed(url: url)) {
                decisionHandler(.cancel)
                return
            }
            let navigationType = navigationAction.navigationType;
            let isPageLoad = parent.hasLoaded == false || navigationType == WKNavigationType.linkActivated ||  (navigationAction.targetFrame?.isMainFrame ?? false)
            if (isPageLoad) {
                let handled = parent.onLoadBegin(url!)
                if (handled) {
                    parent.setLoading(true)
                    decisionHandler(.allow)
                } else {
                    decisionHandler(.cancel)
                }
            } else {
                decisionHandler(.allow)
            }
        }

        func webView(_ webview: WKWebView, didFinish navigation: WKNavigation!) {
            parent.onLoaded()
        }
    }
    
    class WebContentController : WKUserContentController, WKScriptMessageHandler {
        
        let MSG_HANDLER_NAME = "dreadmsg"
        
        var parent: DreadView? = nil
        
        init(_ parent: DreadView) {
            self.parent = parent
            super.init()
        }
        
        required init?(coder: NSCoder) {
            super.init(coder: coder)!
        }
        
        func userContentController(_ userContentController: WKUserContentController, didReceive message: WKScriptMessage) {
            if (parent != nil) {
                if (message.name == MSG_HANDLER_NAME) {
                    if let data = message.body as? [String:AnyObject] {
                        parent!.onMessage(data)
                    }
                }
            }
        }
    }
}
