import SwiftUI
import WebKit

class Session {
    var username: String
    var channels: [String] = []
    init(_ username: String) {
        self.username = username
    }
    func isAuthed() -> Bool {
        return username.count > 0
    }
}

protocol DreadAuthListener {
    func onSessionChange(_ session: Session)
}

protocol DreadRouteListener {
    func onRouteChange(_ route: DreadRoute);
}

class DreadService {
    
    private static var _instance: DreadService? = nil
    static func instance() -> DreadService {
        if (_instance == nil) {
            _instance = DreadService()
        }
        return _instance!;
    }
    
    static let SESSION_NO_AUTH = Session("")
    
    var session: Session = SESSION_NO_AUTH
    var authListeners = MutableSet<DreadAuthListener>()
    var routeListeners = MutableSet<DreadRouteListener>()
    var routes = Dictionary<DreadFrame, URL>()
    @Published var route: DreadRoute
    @Published var frame: DreadFrame
    
    static let TAB_COUNT = 5
    var tabs: [DreadLink] = [
        Links.LINK_HOME,
        Links.LINK_CHANNELS,
        Links.LINK_CREATE_POST,
        Links.LINK_INBOX,
        Links.LINK_PROFILE
    ]
    
    init() {
        route = Routes.HOME_ROUTE
        frame = Routes.HOME_ROUTE.frame
        for tab in tabs {
            routes[tab.frame] = Routes.parseUrl(tab.link)
        }
        navigate(route)
    }
    
    func setSession(_ sessionJson: String?) {
        if (sessionJson == nil) {
            return
        }
        let newSession = type(of: self).parseSession(json: sessionJson)
        if (session.username != newSession.username) {
            session = newSession
            for listener in authListeners.values {
                listener.onSessionChange(session)
            }
        }
    }
    
    func navigate(_ route: DreadRoute) {
        if (route == self.route) {
            return
        }
        if (Routes.isDisallowed(url: route.url)) {
            return
        }
        routes[route.frame] = route.url
        self.route = route
        self.frame = route.frame
        if (route.frame == DreadFrame.POST && route.url.pathComponents.count > 2) {
            let channel = route.url.pathComponents[2]
            routes[DreadFrame.CHANNEL] = Routes.parseUrl("/r/" + channel)
        }
        for listener in routeListeners.values {
            listener.onRouteChange(route)
        }
    }
    
    func getFrameRoute(_ frame: DreadFrame) -> DreadRoute? {
        let url = routes[frame]
        if (url != nil) {
            return Routes.parseRoute(url?.absoluteString)
        }
        return nil
    }
    
    func getFrameLink(_ frame: DreadFrame) -> String {
        let route = getFrameRoute(frame)
        if (route != nil) {
            return route!.url.absoluteString
        }
        return "/"
    }
    
    func getInitialURL(_ frame: DreadFrame) -> URL {
        for tab in tabs {
            if (tab.frame == frame) {
                return URL(string: tab.link)!
            }
        }
        return URL(string: Routes.HOME_URL)!
    }
    
    static func parseSession(json: String?) -> Session {
        if (json != nil && json!.isEmpty == false) {
            do {
                let jsonData = json!.data(using: .utf8)
                let jsonObject = try JSONSerialization.jsonObject(with: jsonData!, options: [])
                if let jsonDict = jsonObject as? [String: Any],
                   let userDict = jsonDict["user"] as? [String: Any],
                   let channelsArray = jsonDict["channels"] as? [String] {
                    if let userName = userDict["name"] as? String {
                        let session = Session(userName)
                        for channel in channelsArray {
                            session.channels.append(channel)
                        }
                        return session
                    }
                } else {
                    print("Invalid JSON data format.")
                }
            } catch {
                print("Error parsing JSON data: \(error)")
            }
        }
        return SESSION_NO_AUTH
    }
}
