#!/usr/bin/env python3

import os
import sys
import subprocess
from datetime import datetime

FROM_YEAR = 2020

try:
	from mutagen import File
	from mutagen.easyid3 import EasyID3
	from mutagen.oggvorbis import OggVorbis
	from mutagen.flac import FLAC
except ImportError:
	print('Mutagen library not found. Installing Mutagen...')
	subprocess.check_call([sys.executable, '-m', 'pip', 'install', 'mutagen'])
	print('Mutagen has been installed.')
	os.execv(sys.executable, ['python'] + sys.argv)
	exit()

def get_audio_date(file_path):
	file_extension = os.path.splitext(file_path)[1].lower()
	if file_extension == '.mp3':
		audio = EasyID3(file_path)
	elif file_extension == '.ogg':
		audio = OggVorbis(file_path)
	elif file_extension == '.flac':
		audio = FLAC(file_path)
	else:
		return None
	date_str = audio.get('date', [None])[0]
	if date_str:
		date = None
		try:
			date = datetime.strptime(date_str, '%Y-%m-%d')
		except ValueError:
			try:
				date = datetime.strptime(date_str, '%Y')
			except ValueError:
				pass
		if date is not None and date.year > FROM_YEAR:
			return date
	return None

def compare_and_update_date(file_path):
	modified_timestamp = os.path.getmtime(file_path)
	modified_date = datetime.fromtimestamp(modified_timestamp)
	tag_date = get_audio_date(file_path)
	if tag_date is not None:
		if tag_date.year <= modified_date.year:
			modified_date = tag_date
		else:
			tag_date = modified_date
		audio = File(file_path, easy=True)
		audio['date'] = tag_date.strftime('%Y-%m-%d')
		audio.save()
	os.utime(file_path, (modified_date.timestamp(), modified_date.timestamp()))

def process_folder(folder_path):
	for root, dirs, files in os.walk(folder_path):
		dirs[:] = [d for d in dirs if not d.startswith('.')]
		for file in files:
			file_path = os.path.join(root, file)
			file_extension = os.path.splitext(file_path)[1].lower()
			if file_extension in ['.mp3', '.ogg', '.flac']:
				compare_and_update_date(file_path)

if __name__ == '__main__':
	if len(sys.argv) < 2:
		print('Please provide the folder path as a command line argument.')
		exit()
	folder_path = sys.argv[1]
	process_folder(folder_path)
