class_name Stage
extends Node3D

var player: Player;
var actors: Array[Actor] = []
var crawl_areas: Array[CrawlArea] = []
var cover_spots: Array[CoverSpot] = []


func register(node: Node) -> void:
	if is_instance_of(node, Actor):
		if actors.has(node):
			return
		actors.append(node)
		if is_instance_of(node, Player):
			player = node
	elif is_instance_of(node, CrawlArea):
		crawl_areas.append(node)
	elif is_instance_of(node, CoverSpot):
		cover_spots.append(node)
	
	
func unregister(node: Node) -> void:
	if is_instance_of(node, Actor) and actors.has(node):
		actors.erase(node)
	elif is_instance_of(node, CrawlArea) and crawl_areas.has(node):
		crawl_areas.erase(node)
	elif is_instance_of(node, CoverSpot) and cover_spots.has(node):
		cover_spots.erase(node)
		
		
func find_jump_area(position: Vector3, direction: Vector3, distance: float = INF) -> CrawlArea:
	var nearest = find_area_nearest(position, distance)
	var result: CrawlArea = null
	var result_dot = -1.0
	for area in crawl_areas:
		if area == nearest:
			continue
		var diff = area.global_transform.origin - position
		diff.y = 0
		if diff.length_squared() > distance * distance:
			continue
		var dot = direction.dot(-diff.normalized())
		if dot < result_dot:
			continue
		result = area
		result_dot = dot
	return result
	
	
func find_area_nearest(position: Vector3, max_distance: float = INF) -> CrawlArea:
	var result: CrawlArea = null
	var result_dist = max_distance * max_distance
	for area in crawl_areas:
		var diff = position - area.global_transform.origin
		diff.y = 0
		var dist = diff.length_squared()
		if dist > result_dist:
			continue
		result = area
		result_dist = dist
	return result
	
	
func find_actor_nearest(position: Vector3, faction: Actor.Faction = Actor.Faction.Any) -> Actor:
	return find_actor_in_range(position, Vector3.FORWARD, INF, -1.0, faction)


func find_actor_in_range(position: Vector3, direction: Vector3, max_distance: float, min_dot: float, faction: Actor.Faction = Actor.Faction.Any) -> Actor:		
	var result: Actor = null
	var result_dsq = max_distance * max_distance
	for actor in actors:
		if faction != Actor.Faction.Any and faction != actor.faction:
			continue
		var diff = actor.global_transform.origin - position
		var dsq = diff.length_squared()
		if dsq > result_dsq:
			continue
		result = actor
		result_dsq = dsq
	return result
	
	
func find_cover_nearby(position: Vector3, min_distance: float, max_distance: float) -> Array[CoverSpot]:
	var result: Array[CoverSpot] = []
	var dsqs: Array[float] = []
	for cover_spot in cover_spots:
		var diff = cover_spot.global_transform.origin - position
		diff.y = 0.0
		var dsq = diff.length_squared()
		if dsq < min_distance * min_distance:
			continue
		if dsq > max_distance * max_distance:
			continue
		var index := -1
		for i in range(dsqs.size()):
			if dsqs[i] > dsq:
				index = i
				break
		if index >= 0:
			result.insert(index, cover_spot)
			result.insert(index, dsq)
		else:
			result.append(cover_spot)
			dsqs.append(dsq)
	return result
