class_name Enemy
extends Actor

enum State {
	Searching,
	Covering,
	Attacking
}

const FIRE_DISTANCE := 5.0
const MOVE_SPEED := 4.0

var weapon: Weapon
var state := State.Searching
var state_time := 0.0
var target: Actor = null
var fire_timer := 0.0
var search_timer := 0.0
var cool_timer := 0.0
var cover: Node3D = null
var cover_timer := 0.0
var last_position := Vector3.ZERO
var is_moving := false


func _init():
	faction = Faction.Evil
	
	
func _ready():
	weapon = get_node("Hand/Bottle") as Weapon
	weapon.set_holding(true)
	state = start_cover()
	
	
func _physics_process(delta: float):
	var prev_state = state
	state = update_state(delta)
	if state != prev_state:
		state_time = 0.0
	is_moving = (transform.origin - last_position).length_squared() > 0.25
	last_position = transform.origin
		

func update_state(delta: float) -> State:
	match state:
		State.Covering:
			return update_cover(delta)
		State.Attacking:
			return update_attack(delta)
	return update_search(delta)


func start_search():
	target = null
	search_timer = randf_range(0.2, 0.5)
	return State.Searching


func update_search(delta: float) -> State:
	search_timer -= delta
	if search_timer < 0.0:
		target = stage.find_actor_in_range(global_transform.origin, global_transform.basis.z, 10.0, 0.0, Actor.Faction.Good)
		if target:
			return start_attack()
	return State.Searching
	
	
func start_cover():
	cover_timer = randf_range(3.0, 7.0)
	target = stage.find_actor_in_range(global_transform.origin, global_transform.basis.z, 10.0, 0.0, Actor.Faction.Good)
	var covers = stage.find_cover_nearby(global_transform.origin, 1.0, 10.0)
	cover = null
	for c in covers:
		if target.can_see(c):
			cover = c
	if not cover:
		var cover_count = covers.size()
		if cover_count > 0:
			var cover_index = randi_range(0, cover_count - 1)
			cover = covers[cover_index]
	if cover:
		return State.Covering
	else:
		cover = null
		return State.Searching
		

func update_cover(delta: float) -> State:
	var in_cover = move_to(cover.global_transform.origin, delta)
	if not is_moving:
		turn_towards(target.global_transform.origin, delta)
		cover_timer -= delta
		if cover_timer <= 0.0:
			return start_cover()
	return State.Covering
	
	
func start_attack():
	fire_timer = randf_range(0.3, 0.7)
	return State.Attacking
	
	
func update_attack(delta: float) -> State:
	var diff = target.global_transform.origin - global_transform.origin
	var weapon_angles = weapon.transform.basis.get_euler()
	weapon_angles.x = acos(diff.length()) + PI * 0.05
	weapon.fire()
	turn_towards(target.global_transform.origin, delta)
	move_to_fire_spot(target.global_transform.origin, delta)
	fire_timer -= delta
	if fire_timer < 0.0:
		return start_search()
	return State.Attacking
	
	
func turn_towards(position: Vector3, delta: float):
	var diff = position - global_transform.origin
	diff.y = 0
	var dir = diff.normalized()
	var rot_to = transform.basis.looking_at(-diff)
	transform.basis = rot_to.slerp(rot_to, Maths.dease(delta, 0.1))
	
	
func move_to_fire_spot(position: Vector3, delta: float) -> bool:
	var diff = position - global_transform.origin
	diff.y = 0
	diff = diff.limit_length(FIRE_DISTANCE)
	var target_pos = global_transform.origin + diff
	return move_to(target_pos, delta)
	
	
func move_to(position: Vector3, delta: float) -> bool:
	var diff = position - global_transform.origin
	diff.y = 0
	var move = (diff * 10.0).limit_length(MOVE_SPEED)
	if move.length_squared() > 0.25:
		velocity = move
		move_and_slide()
		turn_towards(position, delta)
		return false
	else:
		velocity = Vector3.ZERO
		return true
