from unittest import result
from xml.dom.minidom import Element
from xml.etree import ElementTree
from argparse import ArgumentParser
import os
import math
from pathlib import Path

FLIPH  = 0x80000000
FLIPV  = 0x40000000
FLIPD  = 0x20000000
ROT90 = FLIPH | FLIPD
ROT180 = FLIPH | FLIPV
ROT270 = FLIPV | FLIPD

TILE_MAPPING = [
    0, 1,
    1, 0,
    2, 2,
    3, 3,
    4, 4,
    5, 5,

    8, 8 | ROT180,
    9, 8 | ROT270,
    16, 8 | ROT90,
    17, 8,

    10, 9 | ROT180,
    11, 9 | ROT270,
    18, 9 | ROT90,
    19, 9,

    12, 10 | ROT180,
    13, 10 | ROT270,
    20, 10 | ROT90,
    21, 10,
    
    14, 11 | ROT180,
    15, 11 | ROT270,
    22, 11 | ROT90,
    23, 11,

    24, 25 | ROT180,
    25, 25 | ROT270,
    32, 25 | ROT90,
    33, 25,

    26, 18 | ROT180,
    27, 18 | ROT270,
    34, 18 | ROT90,
    35, 18,

    28, 17 | ROT180,
    29, 17 | ROT270,
    36, 17 | ROT90,
    37, 17,

    40, 15,
    48, 23,
    56, 31,

    57, 15 | ROT180,
    49, 23 | ROT180,
    41, 31 | ROT180,

    52, 15 | ROT90,
    51, 23 | ROT90,
    50, 31 | ROT90,

    42, 15 | ROT270,
    43, 23 | ROT270,
    44, 31 | ROT270
]


def transform(elem: Element, forwards: bool, width: int, height: int):
    results = []
    for x in range(width):
        for y in range(height):
            results.append("2")
    s = elem.text
    values = s.split(",")
    for x in range(width):
        for y in range(height):
            i = x + width * y
            try:
                v = int(values[i]) - 1
            except:
                continue
            keyI = -1
            start = 0 if forwards else 1
            for i in range(start, len(TILE_MAPPING), 2):
                if TILE_MAPPING[i] == v:
                    keyI = i
                    break
            if keyI < 0:
                continue
            keyI = math.floor(keyI / 2) * 2
            valueI = TILE_MAPPING[keyI + 1 if forwards else keyI]
            results[x + width * y] = str(valueI + 1)
    elem.text = ",".join(results)


def process_file(file_path, forwards):
    tree = ElementTree.parse(file_path)
    root = tree.getroot()
    layer = root.find(".//layer")
    width = int(layer.attrib["width"])
    height = int(layer.attrib["height"])
    data = layer.find(".//data")
    transform(data, forwards, width, height)
    out_dir = os.path.join(os.path.dirname(file_path), "output")
    if not os.path.exists(out_dir):
        os.mkdir(out_dir)
    out_path = os.path.join(out_dir, os.path.basename(file_path))
    tree.write(out_path)


def process_folder(folder_path, forwards):
    if not os.path.exists(folder_path):
        raise "Folder path %s doesn't exist!" % folder_path

    if not os.path.isdir(folder_path):
        raise "Path is not a directory!"

    for file in Path(folder_path).glob("*.tmx"):
        process_file(os.path.join(folder_path, file.name), forwards)


def main():
    parser = ArgumentParser(description = "Rescale some maps")
    parser.add_argument("folder", type=str, help="Folder to scan")
    parser.add_argument("forwards", type=bool, help="output reoriented")
    args = parser.parse_args()
    process_folder(args.folder, args.forwards)

main()