#!/usr/bin/env python3

import subprocess
import shutil
import os
import time
import platform
from argparse import ArgumentParser
import subprocess


DEFAULT_COMPILER = "gcc"
DEFAULT_COMPILER_WINDOWS = "msvc"


def main():
    parser = ArgumentParser()
    parser.add_argument("--dist", help="dist build", action='store_true')
    parser.add_argument("--run", help="run after build", action='store_true')
    parser.add_argument("--debug", help="compile as debug", action='store_true', default=False)
    parser.add_argument("--compiler", help="compiler msvc|gcc", default='')
    args = parser.parse_args()

    cwd = os.path.realpath(os.path.dirname(__file__) + "/../")
    dir_bin = os.path.realpath(cwd + "/bin")
    dir_dist = os.path.realpath(cwd + "/dist")
    if os.path.exists(dir_dist + "/game_sword.exe"):
        os.remove(dir_dist + "/game_sword.exe")
    if os.path.exists(dir_dist):
        shutil.rmtree(dir_dist)

    rebuild = args.dist
    if rebuild:
        dir_obj = os.path.realpath(cwd + "/obj/Release")
        if os.path.exists(dir_obj):
            shutil.rmtree(dir_obj)
        print("Building %s" % cwd)
    else:
        print("Incremental build %s" % cwd)

    print("Copying dist files...")
    shutil.copytree(dir_bin, dir_dist, ignore=shutil.ignore_patterns("*.exe", "*.sav", "settings.dat", "*.pdb"))

    compiler = args.compiler
    if not compiler:
        compiler = DEFAULT_COMPILER
        if platform.system().lower().startswith("win"):
            compiler = DEFAULT_COMPILER_WINDOWS

    if compiler == 'msvc':
        print("Building with msbuild...")
        result = subprocess.run([
            "msbuild", 
            "./dess.sln", 
            "-t:Rebuild" if rebuild else "-t:Build", 
            "-p:Configuration=%s" % ("Debug" if args.debug else "Release")
        ], cwd=cwd)
        if result.returncode != 0:
            print("Build failed")
            exit(1)
    else:
        print("Building with make...")
        result = subprocess.run(["premake5", "gmake2"], cwd=cwd)
        if rebuild:
            result = subprocess.run(["make", "clean"], cwd=cwd)
        result = subprocess.run([
            "make", 
            "game_sword",
            "config=debug" if args.debug else "config=release"
        ], cwd=cwd)
        if result.returncode != 0:
            print("Build failed")
            exit(1)

    print("Build succeeded")
    
    time.sleep(1)
    shutil.copyfile(dir_bin + "/game_sword.exe", dir_dist + "/game_sword.exe")

    if args.run:
        result = subprocess.run([dir_dist + "/game_sword.exe"], shell=True, cwd=dir_dist)
        if result.returncode != 0:
            print("Exited with code %i" % result.returncode)

    if args.dist:
        version = "0.0"
        with open(dir_dist + "/data/version.txt") as f:
            version = f.read()
        if args.debug:
            version = version.strip() + "-debug"
        zip_path = "build/sword-%s" % version.strip()
        shutil.make_archive(zip_path, "zip", dir_dist)

main()