#ifndef _UI_OPTIONS_H
#define _UI_OPTIONS_H

#include <dessint.h>
#include <desslist.h>
#include "Entity.h"
#include <Textbox.h>
#include <Material.h>
#include <Sprite.h>
#include <RenderWindow.h>
#include <string.h>
#include <Input.h>
#include <SoundPlayer.h>
#include <Maths.h>

#define OPTION_TEXT_LENGTH 16
#define OPTIONS_PER_PAGE 6

struct TitleOption
{
    u8 selection;
    char text[OPTION_TEXT_LENGTH];
};

class IOptionHandler
{
public:
    virtual void OptionSelect(u8 index) = 0;
    virtual void OptionBack() {}
};

class UIOptions : public Entity
{
private:
    Ref<Material> uiMat;
    Textbox txtTitle;
    Sprite spriteSelector;
    u8 firstIndex = 0;
    u8 selectedIndex = 0;
    bool isDirty = false;
    bool isStickDown = false;
    char title[OPTION_TEXT_LENGTH];
    std::vector<TitleOption> options;
    Textbox optionTxts[OPTIONS_PER_PAGE];

public:
    IOptionHandler * handler = NULL;

    void Added()
    {
        uiMat = Material::Load("mats/entities-px.png", "shaders/ui.dsh");

        txtTitle.SetAlignment(Vector2(0.0f, 0.0f));
        txtTitle.SetFontSize(24);
        txtTitle.SetFont("fonts/title");
        txtTitle.Added();
        
        spriteSelector.Load(uiMat);
        spriteSelector.SetSize(32, 32);
        spriteSelector.SetTile(41, 8, 8);

        for (u8 i=0; i<OPTIONS_PER_PAGE; i++)
        {
            optionTxts[i].SetAlignment(Vector2(0.0f, 0.0f));
            optionTxts[i].SetFontSize(18);
            optionTxts[i].SetFont("fonts/pixel");
            optionTxts[i].Added();
        }

        UpdateElements(0.0f);
    }

    void OnEnabled()
    {
        isDirty = true;
    }

    void SetTitle(const char * newTitle)
    {
        strncpy(title, newTitle, OPTION_TEXT_LENGTH);
        isDirty = true;
    }    

    void AddOption(TitleOption option)
    {
        options.push_back(option);
        isDirty = true;
    }

    void SetOption(u8 index, TitleOption option)
    {
        options[index] = option;
        isDirty = true;
    }

    void ClearOptions()
    {
        if (options.size() < 1)
            return;

        options.clear();
        selectedIndex = 0;
        firstIndex = 0;
    }

    void Update(float dt)
    {
        bool wasStickDown = isStickDown;
        Vector2 stick = Input::GetStick(InputStick_LEFT);
        isStickDown = Maths::LengthSq(stick) > 0.4f;
        if (isStickDown && wasStickDown == false)
        {
            u8 optionsCount = options.size();
            if (Maths::Abs(stick.x) > Maths::Abs(stick.y))
            {
                if (stick.x < 0.0f)
                {
                    if (firstIndex > 0)
                    {
                        firstIndex -= OPTIONS_PER_PAGE;
                        isDirty = true;
                    }
                    else
                    {
                        if (handler != NULL)
                        {
                            handler->OptionBack();
                        }
                    }
                }
                if (stick.x > 0.0f && firstIndex < optionsCount - OPTIONS_PER_PAGE)
                {
                    firstIndex += OPTIONS_PER_PAGE;
                    isDirty = true;
                }
                if (isDirty)
                {
                    SoundPlayer::Inst().Play("sfx/click.wav");
                    selectedIndex = 0;
                }
            }
            else
            {
                u8 displayCount = optionsCount - firstIndex;
                if (displayCount > OPTIONS_PER_PAGE)
                {
                    displayCount = OPTIONS_PER_PAGE;
                }
                selectedIndex = (selectedIndex + displayCount + (stick.y > 0.0f ? 1 : -1)) % displayCount;
                SoundPlayer::Inst().Play("sfx/click.wav");
            }
        }
        
        if (isDirty)
        {
            isDirty = false;
            UpdateElements(dt);
        }
        
        Textbox * selectedTB = &optionTxts[selectedIndex];
        spriteSelector.transform.SetPosition(Vector3(
            selectedTB->transform.position.x - 50.0f, 
            selectedTB->transform.position.y - 8.0f, 
            0.0f
        ));
                
        if (Input::WasPressed(InputButton_A) || Input::WasKeyPressed(SDL_SCANCODE_RETURN))
        {
            Select(options[selectedIndex + firstIndex].selection);
            SoundPlayer::Inst().Play("sfx/confirm.wav");
        }
    }

    void UpdateElements(float dt)
    {
        RenderWindow * window = RenderWindow::Current();

        Vector3 pos(window->GetViewportWidth() * 0.5f, 60, 0);   

        txtTitle.SetText(title);
        txtTitle.transform.SetPosition(pos);
        txtTitle.Update(dt);

        pos.y += 50;

        u8 optionsCount = options.size() - firstIndex;
        for (u8 i=0; i<OPTIONS_PER_PAGE; i++)
        {
            if (i >= optionsCount)
                continue;
            optionTxts[i].SetText(options[firstIndex + i].text);
            optionTxts[i].transform.SetPosition(pos);
            optionTxts[i].Update(dt);
            pos.y += 35;
        }
    }

    void Select(u8 index)
    {
        if (handler != NULL)
        {
            handler->OptionSelect(index);
        }
    }

    void Render()
    {
        RenderWindow * window = RenderWindow::Current();

        RenderMode prevMode = window->GetRenderMode();
        window->SetRenderMode(RenderMode_Depthless);

        txtTitle.Render();
        u8 optionsCount = options.size() - firstIndex;
        if (optionsCount > OPTIONS_PER_PAGE)
        {
            optionsCount = OPTIONS_PER_PAGE;
        }
        for (u8 i=0; i<optionsCount; i++)
        {
            optionTxts[i].Render();
        }
        spriteSelector.Render();

        window->SetRenderMode(prevMode);
    }
};

#endif