#ifndef _SCREEN_TITLE_H
#define _SCREEN_TITLE_H

#include "Screen.h"
#include <RenderWindow.h>
#include <Game.h>
#include <SwordGame.h>
#include <Material.h>
#include <desslist.h>
#include <TileMap.h>
#include <Camera.h>
#include <MapEntityFactory.h>
#include <MusicPlayer.h>
#include <FileSystem.h>
#include <string>
#include <FileSystem.h>
#include "UIOptions.h"
#include <Sprite.h>
#include "UIOptionsMenuCtrl.h"
#include "UIProgress.h"
#include "Save.h"
#include <loaders/MapPreloader.h>

enum TitleSelection
{
    TitleSelection_Continue,
    TitleSelection_ContinueConfirm,
    TitleSelection_New,
    TitleSelection_NewConfirm,
    TitleSelection_Load,
    TitleSelection_Options,
    TitleSelection_Title,
    TitleSelection_Credits,
    TitleSelection_Sound,
    TitleSelection_Music,
    TitleSelection_Cooked,
    TitleSelection_Exit,
    TitleSelection_SaveOption = 16
};

struct SaveName
{
    char text[16];
};

class ScreenTitle : public Screen, public IOptionHandler
{
private:
    TileMap * map;
    MapEntityFactory entityFactory;
    Camera cam;
    UIOptions uiOptions;
    Sprite spriteLogo;
    std::vector<SaveName> saves;
    float time = 0.0f;
    u8 saveSelectI = 0;
    bool hasSave = false;
    UIOptionsMenuCtrl * optionsCtrl = NULL;
    UIProgress uiProgress;

public:
    void Init()
    {
        RenderWindow * window = RenderWindow::Current();

        MapPreloader * preloader = MapPreloader::Current();
        if (preloader == NULL)
        {
            SwordGame::Inst().PreloadTitleMap();
        }
        map = MapPreloader::Current()->GetMap();
        map->renderQueue = -100;
        AddEntity(map);
        map->Added();

        Save::LoadFile();
        Missions::Inst().Load();

        spriteLogo.Load("mats/logo.png");
        spriteLogo.SetSize(167, 128);
        spriteLogo.transform.SetPosition(Vector3(50, 50, 0));

        cam.transform.LookDirection(Vector3(0, -10, -7));
        cam.SetCurrent();

        uiOptions.handler = this;
        AddEntity(&uiOptions);
        AddEntity(&spriteLogo);
        AddEntity(&uiProgress);

        uiProgress.shortMode = true;
        uiProgress.transform.position = window->GetViewportPos(0.55f, 0.7f);

        FindLoads();
        ToTitleOptions();

        if (MusicPlayer::Inst().IsPlaying() == false)
        {
            MusicPlayer::Inst().Play("menu");
        }
    }
    
    void FindLoads()
    {
        hasSave = false;
        DirScan dir = FileSystem::DirOpen("");
        while (const char * filename = FileSystem::DirRead(dir))
        {
            if (FileSystem::IsExtenstion(filename, "sav") == false)
                continue;

            SaveName save;
            strncpy(save.text, filename, 16);
            if (FileSystem::IsExtenstion(save.text, "sav") == false)
                continue;

            if (HashEqual("Save_1.sav", filename))
            {
                hasSave = true;
            }

            saves.push_back(save);
        }
    }

    void ClearOptions()
    {
        uiOptions.ClearOptions();
        uiProgress.SetVisible(false);
    }

    void ToTitleOptions()
    {
        ClearOptions();
        uiOptions.SetTitle("MAIN MENU");
        if (hasSave)
        {
            uiOptions.AddOption(TitleOption { 
                TitleSelection_Continue,
                "Continue"
            });
        }
        uiOptions.AddOption(TitleOption { 
            TitleSelection_New,
            "New Game"
        });
#ifdef DEBUG
        uiOptions.AddOption(TitleOption { 
            TitleSelection_Load,
            "Load Select"
        });
#endif
        uiOptions.AddOption(TitleOption { 
            TitleSelection_Options,
            "Options"
        });
        uiOptions.AddOption(TitleOption { 
            TitleSelection_Cooked,
            "More Games"
        });
        uiOptions.AddOption(TitleOption { 
            TitleSelection_Exit,
            "Exit"
        });
    }

    void ToOptionsOptions()
    {
        ClearOptions();
        optionsCtrl = new UIOptionsMenuCtrl(&uiOptions);
    }

    void ToChapterSelectOptions()
    {
        ClearOptions();
        PresentChapters(0);
    }

    void PresentChapters(u8 startIndex)
    {
        uiOptions.ClearOptions();
        saveSelectI = startIndex;
        char c;
        u8 j;
        for (u8 i=0; i<saves.size(); i++)
        {
            TitleOption option;
            option.selection = TitleSelection_SaveOption + i;
            for (j = 0; j < OPTION_TEXT_LENGTH; j++)
            {
                c = saves[i].text[j];
                if (c == '\0' || c == '.')
                    break;
                if (c == '_' || c == '-')
                    c = ' ';
                option.text[j] = c;
            }
            if (j < OPTION_TEXT_LENGTH - 1)
            {
                option.text[j] = '\0';
            }
            uiOptions.AddOption(option);
        }
    }

    void ToNewGameOptions()
    {
        ClearOptions();
        uiProgress.SetVisible(true);
        uiOptions.SetTitle("START OVER?");
        uiOptions.AddOption(TitleOption { 
            TitleSelection_Title,
            "No (Back)"
        });
        uiOptions.AddOption(TitleOption { 
            TitleSelection_NewConfirm,
            "Yes (Erase)"
        });
    }

    void ToContinueOptions()
    {
        ClearOptions();
        uiProgress.SetVisible(true);
        uiOptions.SetTitle("CONTINUE");
        uiOptions.AddOption(TitleOption { 
            TitleSelection_ContinueConfirm,
            "Continue Game"
        });
        uiOptions.AddOption(TitleOption { 
            TitleSelection_Title,
            "Back To Menu"
        });
    }

    void Update(float dt)
    {
        if (map->IsLoading())
        {
            map->LoadContinue();
        }
        Screen::Update(dt);
        time += dt;
        Vector3 camPos = map->GetPosition(map->data.width / 2, map->data.depth / 2);
        camPos.y = 17.0f;
        camPos.x += sinf(time * 0.2f) * (float)(map->data.width - 7) * 0.5f;
        camPos.z += cosf(time * 0.2f) * (float)(map->data.depth - 5) * 0.5f + 6.0f;
        cam.transform.SetPosition(camPos);
    }

    void OptionBack()
    {
        ToTitleOptions();
    }

    void OptionSelect(u8 selection)
    {
        switch (selection)
        {
            case TitleSelection_Continue: 
                ToContinueOptions();
                break;

            case TitleSelection_ContinueConfirm:
                SwordGame::Inst().LoadGame();
                break;

            case TitleSelection_New: 
            {
                if (hasSave)
                {
                    ToNewGameOptions();
                }
                else
                {
                    SwordGame::Inst().NewGame();
                }
                break;
            }

            case TitleSelection_NewConfirm:
                SwordGame::Inst().NewGame();
                break;

            case TitleSelection_Load: 
                ToChapterSelectOptions();
                break;

            case TitleSelection_Title: 
                ToTitleOptions();
                break;

            case TitleSelection_Cooked: 
                SwordGame::Inst().Cooked();
                break;

            case TitleSelection_Credits: 
                SwordGame::Inst().Credits();
                break;

            case TitleSelection_Options:
                ToOptionsOptions();
                break;
                
            case TitleSelection_Sound:
                SwordGame::Inst().Stop();
                break;

            case TitleSelection_Exit:
                SwordGame::Inst().Stop();
                break;

            default:
            {
                if (selection > TitleSelection_SaveOption)
                {
                    SelectChapterOption(selection - TitleSelection_SaveOption);
                }
            }
        }
    }

    void SelectChapterOption(u8 index)
    {
        SwordGame::Inst().LoadSelect(saves[index].text);
    }
};

#endif