#include <ui/GameUI.h>
#include <RenderWindow.h>
#include <SoundPlayer.h>
#include <Maths.h>
#include <Actor.h>

GameUI * GameUI::current = NULL;

GameUI * GameUI::Current() 
{ 
    return current; 
}

GameUI::GameUI()
{
    current = this;
    useCulling = false;
}

void GameUI::Added()
{
    txtMessage.Added();
    txtMessage.SetAlignment(Vector2(0.5f, 1.0f));
    txtMessage.SetFontSize(16);
    txtPromptConfirm.Added();
    txtPromptConfirm.SetAlignment(Vector2(1.0f, 1.0f));
    txtPromptConfirm.SetFontSize(16);
    txtPromptConfirm.SetColor(Vector3(0.7f, 0.9f, 0.7f));
    txtPromptCancel.Added();
    txtPromptCancel.SetAlignment(Vector2(1.0f, 1.0f));
    txtPromptCancel.SetFontSize(16);
    txtPromptCancel.SetColor(Vector3(0.9f, 0.7f, 0.7f));
    txtName.Added();
    txtName.SetAlignment(Vector2(0.5f, 1.0f));
    txtName.SetFontSize(16);
    txtName.SetColor(Vector3(0.8f, 0.8f, 0.7f));
    txtChicken.Added();
    txtChicken.SetAlignment(Vector2(1.0f, 1.0f));
    txtChicken.SetFontSize(16);
    txtChicken.SetText("0");

    uiMat = Material::Load("mats/entities-px.png", "shaders/ui.dsh");
    for (u8 i = 0; i < UI_HEARTS_MAX; i++)
    { 
        spriteHearts[i].Load(uiMat);
        spriteHearts[i].SetSize(32, 32);
        spriteHearts[i].SetRect((u8)1, (u8)4, (u8)8, (u8)8);
        spriteHearts[i].SetEnabled(true);
    }

    spriteHolding.Load(uiMat);
    spriteHolding.SetSize(32, 32);
    spriteHolding.SetEnabled(false);

    spriteChicken.Load(uiMat);
    spriteChicken.SetSize(32, 32);
    spriteChicken.SetRect((u8)0, (u8)4, (u8)8, (u8)8);

    spriteChicken.SetEnabled(false);
    txtChicken.SetEnabled(false);

    renderQueue = 100;
}

void GameUI::SetMessage(const char * title, const char * message, float time, MessageType messageType)
{
    txtPromptConfirm.SetEnabled(false);
    txtPromptCancel.SetEnabled(false);
    txtName.SetEnabled(true);
    txtMessage.SetEnabled(true);
    txtMessage.SetMaxDisplayLength(1);
    isWriting = true;
    if (messageType == MessageType_Silent)
    {
        nextSoundTime = 100.0f;
    }
    else
    {
        nextSoundTime = 0.0f;
    }
    SetText(title, time, &txtName);
    SetText(message, time, &txtMessage);
    isTalking = false;
    if (messageType == MessageType_Confirm)
    {
        messageStatus = MessageStatus_Pending;
    }
    else
    {
        messageStatus = MessageStatus_None;
    }
}

void GameUI::SetTalkMessage(const char * title, const char * message, float time, MessageType messageType)
{
    SetMessage(title, message, time, messageType);
    isTalking = true;
}

void GameUI::SetMessage(const char * message, float time, MessageType messageType)
{
    txtPromptConfirm.SetEnabled(false);
    txtPromptCancel.SetEnabled(false);
    txtName.SetEnabled(false);
    txtMessage.SetEnabled(true);
    txtMessage.SetMaxDisplayLength(1);
    isWriting = true;
    SetText(message, time, &txtMessage);
    if (messageType == MessageType_Silent)
    {
        nextSoundTime = 100.0f;
    }
    else
    {
        nextSoundTime = 0.0f;
    }
    messageStatus = MessageStatus_None;
}

void GameUI::SetPrompt(const char * text)
{
    FinishWriting();
    txtPromptConfirm.SetEnabled(true);
    txtPromptCancel.SetEnabled(false);
    SetText(text, 1000.0f, &txtPromptConfirm);
    isTalking = false;
}

void GameUI::SetConfirmPrompt(const char * confirmText, const char * cancelText)
{
    FinishWriting();
    txtPromptConfirm.SetEnabled(true);
    txtPromptCancel.SetEnabled(true);
    SetText(confirmText, 1000.0f, &txtPromptConfirm);
    SetText(cancelText, 1000.0f, &txtPromptCancel);
    isTalking = false;
}

void GameUI::HideMessages()
{
    txtName.SetEnabled(false);
    txtMessage.SetEnabled(false);
    txtPromptConfirm.SetEnabled(false);
    txtPromptCancel.SetEnabled(false);
    isTalking = false;
}

bool GameUI::HasMessage() const
{
    return txtMessage.IsEnabled();
}

void GameUI::SetText(const char * text, float time, Textbox * textbox)
{
    if (text == NULL || text[0] == '\0')
    {
        messageTime = 0.0f;
        textbox->SetEnabled(false);
        return;
    }
    messageTime = time;

    textbox->SetMaxWidth(RenderWindow::Current()->GetViewportWidth() - 100.0f);
    textbox->SetText(text);
    
    displayTime = 0.0f;
}

void GameUI::SetInventory(ActorInventory * inventory)
{
    u8 newHeartCount = inventory->hearts;
    if (newHeartCount > UI_HEARTS_MAX * 2)
        newHeartCount = UI_HEARTS_MAX * 2;

    if (newHeartCount != heartCount)
    {
        heartCount = newHeartCount;
        for (i8 i=0; i<UI_HEARTS_MAX; i++)
        {
            spriteHearts[i].SetEnabled(i * 2 < heartCount);
            u8 hx = i * 2 + 1 == heartCount ? 2 : 1;
            spriteHearts[i].SetRect(hx, (u8)4, (u8)8, (u8)8);
        }
    }

    u8 newHoldItem = (u8)inventory->holdItem;
    if (newHoldItem != holdItem)
    {
        holdItem = newHoldItem;
        if (holdItem == MapEntityType_None)
        {
            spriteHolding.SetEnabled(false);
        }
        else
        {
            spriteHolding.SetTile(MapEntityTypes::ToId((MapEntityType)holdItem), 8, 8);
            spriteHolding.SetEnabled(true);
        }
    }

    u8 newChickenCount = inventory->chicken;
    if (newChickenCount != chickenCount)
    {
        chickenCount = newChickenCount;
        char chickenCountStr[8];
        sprintf(chickenCountStr, "%i", chickenCount);
        txtChicken.SetText(chickenCountStr);
        txtChicken.SetEnabled(chickenCount > 0);
        spriteChicken.SetEnabled(chickenCount > 0);
    }
}

void GameUI::FinishWriting()
{
    if (isWriting)
    {
        isWriting = false;
        txtMessage.SetMaxDisplayLength(UINT16_MAX);
    }
}

void GameUI::Update(float dt)
{
    RenderWindow * window = RenderWindow::Current();
    float width = window->GetViewportWidth();
    float height = window->GetViewportHeight();

    if (txtMessage.IsEnabled())
    {
        txtMessage.transform.SetPosition(Vector3(width * 0.5f, height - 50.0f, 0.0f));
        txtName.transform.SetPosition(Vector3(width * 0.5f, txtMessage.transform.position.y - txtMessage.GetHeight() - 10.0f, 0.0f));
    }
    else if (txtPromptConfirm.IsEnabled())
    {
        txtPromptConfirm.transform.SetPosition(Vector3(width - 30.0f, height - 25.0f, 0.0f));
        if (txtPromptCancel.IsEnabled())
        {
            txtPromptCancel.transform.SetPosition(txtPromptConfirm.transform.position + Vector3(-txtPromptCancel.GetWidth() - 10, 0, 0));
        }
    }

    for (u8 i=0; i<UI_HEARTS_MAX; i++)
    {
        spriteHearts[i].transform.SetPosition(
            Vector3(30 + (i % 4) * 24, 30 + (i / 4) * 24, 0)
        );
    }

    spriteHolding.transform.SetPosition(
        Vector3(30, 60 + (heartCount / 8) * 24, 0)
    );

    spriteChicken.transform.SetPosition(
        Vector3(width - 56, 30, 0)
    );
    txtChicken.transform.SetPosition(spriteChicken.transform.position + Vector3(0, 16, 0));
    
    displayTime += dt;
    if (isWriting)
    {
        u16 maxDisplayLength = u16(displayTime * 40.0f) + 1;
        txtMessage.SetMaxDisplayLength(maxDisplayLength);
        if (maxDisplayLength > txtMessage.GetTextLength())
        {
            isWriting = false;
        }
        if (nextSoundTime <= 0.0f)
        {
            char tsound[16];
            sprintf(tsound, "sfx/talk%i.wav", Maths::RandI(3) + 1);
            SoundPlayer::Inst().Play(tsound);
            nextSoundTime = 0.05f;
        }
        else
        {
            nextSoundTime -= dt;
        }
    }

    if (txtMessage.IsEnabled() && messageTime > 0.0f)
    {
        if (isWriting == false)
        {
            messageTime -= dt;
        }
    }
}

void GameUI::Render()
{
    RenderWindow * window = RenderWindow::Current();
    RenderMode prevMode = window->GetRenderMode();
    window->SetRenderMode(RenderMode_Depthless);
    if (messageTime > 0.0f)
    {
        if (txtPromptConfirm.IsEnabled())
            txtPromptConfirm.Render(); 
        if (txtPromptCancel.IsEnabled())
            txtPromptCancel.Render(); 
        if (txtName.IsEnabled())
            txtName.Render();
        if (txtMessage.IsEnabled())
            txtMessage.Render();
    }
    for (u8 i=0; i<UI_HEARTS_MAX; i++)
    {
        if (spriteHearts[i].IsEnabled())
            spriteHearts[i].Render();
    }
    if (spriteHolding.IsEnabled())
        spriteHolding.Render();
    if (spriteChicken.IsEnabled())
    {
        spriteChicken.Render();
        txtChicken.Render();
    }
    window->SetRenderMode(prevMode);
}