#ifndef _PUSHABLE_PROP_H
#define _PUSHABLE_PROP_H

#include "Prop.h"
#include <Maths.h>
#include <SoundPlayer.h>
#include <CollisionChecker.h>
#include <fx/FXer.h>

class PushableProp : public Prop
{
protected:
    Vector3 pushDirection = Vector3_ZERO;
    Vector3 autoPushDirection = Vector3_ZERO;
    MapPoint lastPoint;
    u16 spacesRemaining = 0;
    u16 spacesTotal = 0;
    bool recentering = true;
    bool wantMove = false;
    float pushForce = 0.0f;
    float timeToPush = 0.1f;
    float speed = 8.0f;
    float respawnTime = 0.0f;
    float noPushTime = 0.0f;
    float moveTime = 0.0f;

public:
    PushableProp(MapEntityType type, u16 spaces) : 
        Prop(type),
        spacesTotal(spaces)
    {}

    void Start()
    {
        radius = 0.8f;
        MapEntity::Start();
        SetCollision(CollisionMode_Resolve, Vector3(1.5f, 1.2f, 1.5f), Vector3(0.0f, 0.6f, 0.0f));
        collider->mass = 3.0f;
        acceleration = 1.0f;
        deceleration = 3.0f;
        useGravity = true;
        canDespawn = false;
        if (map->FlagIsSet(info->flag))
        {
            const MapObject * entity = map->GetEntityById(info->flag);
            if (entity != NULL)
            {
                transform.position = map->GetPosition(entity->x, entity->z, 0.5f, 0.5f);;
            }
        } 
        else
        {
            autoPushDirection = Vector3_ZERO;
            if (info->cmdCount > 0)
            {
                for (u8 i=0; i<info->cmdCount; i++)
                {
                    const MapCmd * cmd = &map->data.cmds[i + info->cmdFirst];
                    if (cmd != NULL && cmd->type == MapCmd_DIR)
                    {
                        autoPushDirection = Directions::GetForward((Direction)cmd->value);
                        break;
                    }
                }
            }
        }
    }

    virtual void Reset()
    {
        Prop::Reset();
        pushDirection = Vector3_ZERO;
        timeToPush = 0.0f;
        pushForce = 0.0f;
        respawnTime = 0.0f;
        SetCollision(CollisionMode_Resolve);
    }

    void Update(float dt)
    {
        if (respawnTime > 0.0f)
        {
            respawnTime -= dt;
            if (respawnTime <= 0.0f)
            {
                Reset();
                transform.position = spawnPos;
                transform.position.y += 15.0f;
            }
            return;
        }
        if (noPushTime > 0.0f)
        {
            noPushTime -= dt;
            pushDirection = Vector3_ZERO;
            pushForce = 0.0f;
        }
        MapEntity::Update(dt); 
        bool recentered = false;
        if (recentering)
        {
            bool moveXGreater = Maths::Abs(moveVelocity.x) > Maths::Abs(moveVelocity.z);
            bool recenterX = isMoving == false || moveXGreater == false;
            bool recenterZ = isMoving == false || moveXGreater;
            recentered = recenterX || recenterZ;

            Vector3 tilePos = map->GetPosition(point.x, point.z);
            float maxAdjustment = dt * 3.0f;
            if (recenterX)
            {
                transform.position.x += Maths::Clamp(tilePos.x - transform.position.x, -maxAdjustment, maxAdjustment);
            }
            if (recenterZ)
            {
                transform.position.z += Maths::Clamp(tilePos.z - transform.position.z, -maxAdjustment, maxAdjustment);
            }
        }
        if (wantMove)
        {
            pushForce += dt;
            wantMove = false;
            if (pushForce > timeToPush)
            {
                DoPush();
            }
        }
        if (isMoving)
        {
            pushForce = 0.0f;
            wantMove = false;
            moveTime += dt;
        }
        else
        {
            moveTime = 0.0f;
        }
        collider->mass = isMoving || recentered ? 2.0f : 3.0f;
        if (isGrounded && Maths::LengthSq(autoPushDirection) > 0.1 && isMoving == false)
        {
            Push(autoPushDirection, true);
        }
        dirMove = pushDirection;
    }
    
    Vector3 GetMoveVelocity(float dt)
    {
        if (lastPoint != point)
        {
            OnSpaceChange();
            spacesRemaining--;
            if (spacesRemaining < 1)
            {
                spacesRemaining = 0;
                isMoving = false;
                pushDirection = Vector3_ZERO;
            }
            lastPoint = point;
        }
        Vector3 move = isMoving ? pushDirection * speed : Vector3_ZERO;
        return move;
    }

    virtual void OnSpaceChange() {}

    void Push(Vector3 direction, bool force = false)
    {
        CollisionChecker * checker = CollisionChecker::Current();
        if (checker != NULL)
        {
            Entity * entityAtPoint = checker->CheckPoint(transform.position + direction * 2.0f + Vector3(0, 0.5f, 0));
            if (Entity::IsActive(entityAtPoint) && entityAtPoint->collider->mode >= CollisionMode_Collide)
            {
                return;
            }
        }
        pushDirection = direction;
        if (force == false && isMoving == false)
        {
            wantMove = true;
            return;
        }
        DoPush();
    }

    void DoPush()
    {
        if (isMoving)
            return;
        isMoving = true;
        lastPoint = point;
        spacesRemaining = spacesTotal;
        if (Maths::LengthSq(autoPushDirection) < 0.1)
        {
            SoundPlayer::Inst().Play("sfx/slide.wav", 0.8f);
        }
    }
    
    void OutOfMap()
    {
        if (respawnTime > 0.0f)
            return;
        respawnTime = 1.0f;
        SetCollision(CollisionMode_None);
    }

    void Render()
    {
        if (respawnTime > 0.0f)
            return;

        Prop::Render();
    }

    void OnCollision(Entity * other)
    {
        if (moveTime > 0.2f)
        {
            MapEntity * me = (MapEntity*)other;
            if (Entity::IsActive(me))
            {
                if (moveTime > 0.5f && MapEntityTypes::IsActor(me->type))
                {
                    me->TakeHit(this, 1);
                }
                wantMove = false;
                pushDirection = Vector3_ZERO;
                recentering = true;
                noPushTime = 0.5f;
            }
            FXer::Fire(this, FXType_Dust, Vector3_UP);
        }
    }
};

#endif