#ifndef _PROP_SWITCH_H
#define _PROP_SWITCH_H

#include <MapEntity.h>
#include <Model3D.h>
#include <MeshRenderer.h>
#include <Material.h>
#include <FileSystem.h>

class PropSwitch : public MapEntity
{
protected:
    Model3D model;
    Ref<Material> materialOn;
    Ref<Material> materialOff;
    MeshRenderer renderer;
    float switchOnTime = 0.0f;
    bool isOn = false;
    bool staysOn = false;
    
public:
    PropSwitch(MapEntityType type = MapEntityType_Switch) : MapEntity(type) {}

    virtual ~PropSwitch()
    {
        Material::Free(materialOn);
        Material::Free(materialOff);
    }

    virtual void Start()
    {
        radius = 0.2f;
        MapEntity::Start();
        SetCollision(CollisionMode_Receive, Vector3(0.2f, 0.5f, 0.2f), Vector3(0, 0.5f, 0));
        useGravity = false;
        isOn = map->FlagIsSet(info->id);
        LoadModel();
    }

    virtual void LoadModel()
    {
        LoadModel(
            "models/switch.iqm",
            "mats/switch_on.png", 
            "mats/switch_off.png"
        );
    }

    virtual void LoadModel(const char * modelPath, const char * textureOn, const char * textureOff, const char * shaderOn = "shaders/standard.dsh", const char * shaderOff = NULL)
    {
        if (shaderOff == NULL)
        {
            shaderOff = shaderOn;
        }
        materialOn = Material::Load(textureOn, shaderOn);
        materialOff = Material::Load(textureOff, shaderOff);
        model.LoadModel(modelPath);
        model.SetMaterial(isOn ? materialOn : materialOff);
    }

    virtual void Update(float dt)
    {
        MapEntity::Update(dt);
        if (staysOn == false && switchOnTime > 0.0f)
        {
            switchOnTime -= dt;
            if (switchOnTime <= 0.0f)
            {
                if (collider->collisions.size() == 0)
                {
                    SetOn(false);
                }
            }
        }
    }

    virtual void Render()
    {
        model.Render(transform.GetMatrix());
    }

    virtual void OnCollision(Entity * other)
    {
        SetOn(true, 0.1f);
    }

    virtual void SetOn(bool nowOn, float time = INFINITY)
    {
        if (nowOn != isOn)
        {
            isOn = nowOn;
            model.SetMaterial(isOn ? materialOn : materialOff);
            map->FlagSet(info->id, isOn);
        }
        if (nowOn)
        {
            switchOnTime = time;
        }
    }

    inline bool IsOn() const { return isOn; }
};

#endif