#ifndef _PROP_LAVA_BALL_H
#define _PROP_LAVA_BALL_H

#include "Prop.h"
#include <Scene.h>
#include <Actor.h>
#include <Maths.h>
#include <fx/Shadow.h>
#include <SoundPlayer.h>

class PropLavaBall : public Prop
{
public:
    float time = 0.0f;
    float airTime = 3.0f;
    float delay = 0.0f;
    Vector3 targetPos = Vector3_ZERO;
    Vector3 axis = Vector3_RIGHT;
    Entity * target = NULL;
    Shadow shadow;
    bool isGrounded = false;

    PropLavaBall() : Prop(MapEntityType_LavaBall), shadow(this) {}

    void Start()
    {
        radius = 0.7f;
        Prop::Start();
        LoadModel("models/lavaball.iqm", "mats/lava.png");
        SetCollision(CollisionMode_Collide, Vector3(radius * 2.0f), Vector3(0, 0.25f, 0));
        useGravity = false;
        canDespawn = false;
        useCulling = false;
    }

    void OnEnabled()
    {
        delay = (sin(transform.position.x * 0.35f) + 1.0f) * 1.2f + (cos(transform.position.z * 0.35f) + 1.0f) * 0.5f;
        SetVisible(false);
    }

    void Fire()
    {
        Entity * target = scene->GetNearestEntity(transform.position, ActorFaction_PLAYER, 15.0f);
        if (target != NULL)
        {
            targetPos = target->transform.position;
        }
        else
        {
            targetPos = spawnPos;
        }
        time = 0.0f;
        airTime = 2.0f;
        transform.position.y = -2.0f;
        transform.SetScale(1.0f);
        SetVisible(true);
        axis = Maths::Normalize(Vector3(Maths::RandFSigned(1.0f), Maths::RandFSigned(1.0f), Maths::RandFSigned(1.0f)));
        isGrounded = false;
        SoundPlayer::Inst().Play("sfx/strike_fire.wav", 0.5f);
    }

    void Update(float dt)
    {
        if (delay > 0.0f)
        {
            delay -= dt;
            if (delay <= 0.0f)
            {
                Fire();
            }
            return;
        }
        if (time < airTime * 0.5f || target == NULL)
        {
            if (target != NULL)
            {
                targetPos = target->transform.position + target->transform.Forwards();
            }
            time += dt;
        }
        else
        {
            time += dt * 0.3f;
        }
        float pc = time / airTime;
        transform.position = Maths::Lerp(spawnPos, targetPos, pc);
        transform.position.y = (1.0f - (pc + pc - 1.0f) * (pc + pc - 1.0f)) * 12.0f;
        float groundY = map->GetHeight(transform.position);
        if (groundY > transform.position.y && isGrounded == false)
        {
            if (groundY > MAP_WATER_HEIGHT)
            {
                isGrounded = true;
                SoundPlayer::Inst().Play("sfx/thud3.wav", 0.3f);
            }
            else
            {
                pc = 2.1f;
            }
        }
        if (isGrounded)
        {
            transform.scale.y = 0.01f;
            transform.scale.x = transform.scale.z = -pc * 0.5f + 1.5f;
            transform.position.y = groundY + (pc - 1.0f) * 0.1f + 0.01f;
            transform.rotation = Quat_IDENTITY;
        }
        else
        {
            transform.Rotate(axis, dt * 5.0f);
        }
        if (pc > 1.6f)
        {
            delay = 1.5f;
            SetVisible(false);
            transform.position.y = -2.0f;
        }
    }

    void OnCollision(Entity * other)
    {
        if (Actor::IsActor(other))
        {
            ((Actor*)other)->TakeHit(NULL, 1);
            delay = 2.0f;
            SetVisible(false);
            transform.position.y = -2.0f;
        }
    }

    void Render()
    {
        shadow.Render();
        Prop::Render();
    }
};

#endif