#include "PropDoor.h"
#include <Maths.h>
#include <Collider.h>
#include <Scene.h>
#include <Actor.h>
#include <MapData.h>
#include <SoundPlayer.h>

void PropDoor::Start()
{
    Prop::Start();
    radius = 0.9f;
    autoOpen = false;

    LoadDoorModel();

    if (info->cmdCount > 0)
    {
        for (u8 i=0; i<info->cmdCount; i++)
        {
            const MapCmd * cmd = &map->data.cmds[info->cmdFirst + i];
            if (cmd->type == MapCmd_UNLOCK)
            {
                unlockFlag = cmd->value;
                if (unlockFlag == 1)
                {
                    autoOpen = true;
                }
            }
        }
    }

    float dir = transform.scale.x > 0.0f ? -1.0f : 1.0f;
    rotClosed = transform.rotation;
    modelOffset = map->data.tileSize * dir * 0.5f * transform.Right();
    SetCollision(CollisionMode_Collide, Vector3(2.0f, 2.0f, 0.6f), Vector3_UP);
    SetOpen(ShouldBeOpen(), true);
    if (isOpen == false && ShouldBeUnlocked() == false)
    {
        isLocked = true;
    }
    collider->mass = 10000.0f;
}

void PropDoor::LoadDoorModel()
{
    LoadModel("models/door.iqm", "mats/woodboard.png");
}

void PropDoor::Update(float dt)
{
    MapEntity::Update(dt);

    bool shouldOpen = ShouldBeOpen();
    if (shouldOpen != isOpen)
    {
        SetOpen(shouldOpen);
    }
    transform.rotation = Maths::Slerp(transform.rotation, rotTo, 0.1f, dt);
}

void PropDoor::Open()
{
    SetOpen(true);
    map->FlagSet(info->id, true);
}

void PropDoor::SetOpen(bool value, bool force)
{
    if (value == isOpen && force == false)
        return;

    isOpen = value;

    if (isOpen)
    {
        isLocked = false;
        collider->mode = CollisionMode_None;

        float direction = 1.0f;
        Entity * player = scene->GetNearestEntity(transform.position, ActorFaction_PLAYER);
        if (player != NULL)
        {
            float front2 = Maths::LengthSq(player->transform.position - transform.position + transform.Forwards());
            float back2 = Maths::LengthSq(player->transform.position - transform.position - transform.Forwards());
            if (front2 < back2)
            {
                direction = -direction;
            }
        }
        rotTo = glm::angleAxis(PI * -0.5f * direction * Maths::Sign(transform.scale.x), Vector3_UP) * rotClosed;

        SoundPlayer::Inst().Play("sfx/door.wav");
    }
    else
    {
        rotTo = rotClosed;
        collider->mode = CollisionMode_Collide;
        if (isLocked && ShouldBeUnlocked())
        {
            SetOpen(true);
        }
    }
}

bool PropDoor::ShouldBeOpen()
{
    if (unlockFlag && map->FlagIsSet(unlockFlag))
        return true;
    return map->FlagIsSet(info->id) || (
        info->flag > 0 && map->FlagIsSet(info->flag)
    ) && ShouldBeUnlocked();
}

bool PropDoor::ShouldBeUnlocked()
{
    return unlockFlag == 0 || map->FlagIsSet(unlockFlag);
}