#include "TiledUtils.h"
#include <MapData.h>
#include <string.h>
#include <dessstrings.h>

const u16 TILE_INFO_MAP[] = {
    0,  TileType_Empty, 0, 0,
    1,  TileType_Solid, 1, 1,
    2,  TileType_Solid, 2, 2,
    3,  TileType_Solid, 3, 3,
    4,  TileType_Solid, 4, 4,
    5,  TileType_Solid, 5, 5,
        
    8,  TileType_CornerNE, 1, 0,
    9,  TileType_CornerNE, 2, 1,
    10, TileType_CornerNE, 3, 2,
    11, TileType_CornerNE, 4, 3,
    12, TileType_CornerNE, 5, 4,
    
    16, TileType_CornerNE, 2, 0,
    17, TileType_CornerNE, 3, 1,
    18, TileType_CornerNE, 4, 2,
    19, TileType_CornerNE, 5, 3,

    24, TileType_CornerNE, 3, 0,
    25, TileType_CornerNE, 4, 1,
    26, TileType_CornerNE, 5, 2,

    32, TileType_CornerNE, 4, 0,
    33, TileType_CornerNE, 5, 1,

    7,  TileType_RampN, 5, 4,
    15, TileType_RampN, 4, 3,
    23, TileType_RampN, 3, 2,
    31, TileType_RampN, 2, 1,

    6,  TileType_Solid,    MAP_LAYER_WATER + 1, 0,
    14, TileType_CornerNE, 1, MAP_LAYER_WATER + 1,
    22, TileType_CornerNE, 2, MAP_LAYER_WATER + 1,
    30, TileType_CornerNE, 3, MAP_LAYER_WATER + 1,
    38, TileType_CornerNE, 4, MAP_LAYER_WATER + 1
};

const u16 TILE_INFO_COUNT = (u16)(sizeof(TILE_INFO_MAP) / sizeof(u16));

const u32 TILED_FLIP_H  = 0x80000000;
const u32 TILED_FLIP_V  = 0x40000000;
const u32 TILED_FLIP_D  = 0x20000000;
const u32 TILED_FLIP_R = 0x10000000;
const u32 TILED_ROT_0   = 0;
const u32 TILED_ROT_90  = TILED_FLIP_H | TILED_FLIP_D;
const u32 TILED_ROT_180 = TILED_FLIP_H | TILED_FLIP_V;
const u32 TILED_ROT_270 = TILED_FLIP_V | TILED_FLIP_D;

u16 TiledUtils::GetTileInfoIndex(u16 inputTile)
{
    for (u16 i=0; i < TILE_INFO_COUNT; i += 4)
    {
        if (TILE_INFO_MAP[i] == inputTile)
            return i;
    }
    return 0;
}

void TurnType(TileType * type, u32 gid, u8 first, u8 addTurns)
{
    u8 turned = *type - first + addTurns;
    if ((gid & TILED_ROT_90) == TILED_ROT_90)
        turned += 1;
    if ((gid & TILED_ROT_180) == TILED_ROT_180)
        turned += 2;
    if ((gid & TILED_ROT_270) == TILED_ROT_270)
        turned += 3;
    *type = (TileType)(first + (turned % 4));
}

void TiledUtils::ExtractGID(u32 gid, u8 * out_type, bool * flipH, bool * flipV, bool * flipD)
{
    if (flipH != NULL)
    {
        *flipH = (gid & TILED_FLIP_H);
    }
    if (flipV != NULL)
    {
        *flipV = (gid & TILED_FLIP_V);
    }
    if (flipD != NULL)
    {
        *flipD = (gid & TILED_FLIP_D);
    }

    *out_type = ((u16)(gid & ~(
        TILED_FLIP_H |
        TILED_FLIP_V |
        TILED_FLIP_D |  
        TILED_FLIP_R
    )) % 64) - 1;
}

void TiledUtils::ExtractTileInfo(const char * gidStr, TileInfo * out)
{
    u32 gid = StringToUint(gidStr);
    ExtractTileInfo(gid, out);
}

void TiledUtils::ExtractTileInfo(u32 gid, TileInfo * out)
{
    u8 type = 0;
    ExtractGID(gid, &type, NULL, NULL, NULL);
    u16 tileInfoIndex = GetTileInfoIndex(type);

    TileType tileType = (TileType)TILE_INFO_MAP[tileInfoIndex + 1];
    out->layer = (u8)TILE_INFO_MAP[tileInfoIndex + 2];
    out->under = (u8)TILE_INFO_MAP[tileInfoIndex + 3];
    switch (tileType)
    {
        case TileType_CornerNE:
            TurnType(&tileType, gid, tileType, 1);
            break;
        case TileType_RampN:
            TurnType(&tileType, gid, tileType, 0);
            break;
    }
    out->type = (TileType)tileType;
}

void TiledUtils::ExtractMusicArgs(const char * s, u16 * out_start, u16 * out_end, u16 * out_repeat)
{
    if (s == NULL || s[0] == '\0')
        return;
    
    u16 args[] = { 0, 0, 0 };
    u8 ai = 0;
    char c;
    for (u8 i=0; i<MAP_NAME_LENGTH; i++)
    {
        c = s[i];
        if (c == '\0')
            break;

        if (c == ' ')
        {
            ai++;
        }

        if (c >= '0' && c <= '9')
        {
            args[ai] = args[ai] * 10 + (u16)(c - '0');
        }
    }
    if (out_start != NULL)
    {
        *out_start = args[0];
    }
    if (out_end != NULL)
    {
        *out_end = args[1];
    }
    if (out_repeat != NULL)
    {
        *out_repeat = args[2];
    }
}

void TiledUtils::ExtractMapCmd(const char * propName, MapCmd * out)
{
    out->type = MapCmd_NONE;
    char cmd[16] = { 0 };
    char arg[16] = { 0 };
    u8 split = 0;
    u8 offset = 0;
    for (u8 i = 0; i < 3; i++)
    {
        if (propName[i] == '\0')
            break;

        if ((propName[i] >= '0' && propName[i] <= '9') || propName[i] == ' ')
        {
            offset++;
            continue;
        }

        break;
    }
    for (u8 i = offset; i<16; i++)
    {
        if (propName[i] == '\0' || propName[i] == '#')
        {
            cmd[i - offset] = '\0';
            break;
        }
        if (split == 0 && propName[i] == ' ')
        {
            cmd[i - offset] = '\0';
            split = i + 1;
            continue;
        }
        if (split > 0)
        {
            arg[i - split] = propName[i];
            arg[i - split + 1] = '\0';
        }
        else
        {
            cmd[i - offset] = propName[i];
            cmd[i - offset + 1] = '\0';
        }
    }
    if (HashEqual(cmd, "say") || HashEqual(cmd, "message"))
    {
        out->type = MapCmd_SAY;
    }
    else if (HashEqual(cmd, "then"))
    {
        out->type = MapCmd_THEN;
    }
    else if (HashEqual(cmd, "if"))
    {
        out->type = MapCmd_IF;
        out->param = StringToUint(arg);
    }
    else if (HashEqual(cmd, "if$"))
    {
        out->type = MapCmd_IF_MONEY;
        out->param = StringToUint(arg);
    }
    else if (HashEqual(cmd, "ifnot"))
    {
        out->type = MapCmd_IFNOT;
        out->param = StringToUint(arg);
    }
    else if (HashEqual(cmd, "prompt"))
    {
        out->type = MapCmd_PROMPT;
        out->param = StringToUint(arg);
    }
    else if (HashEqual(cmd, "unlock"))
    {
        out->type = MapCmd_UNLOCK;
    }
    else if (HashEqual(cmd, "map"))
    {
        out->type = MapCmd_MAP;
    }
    else if (HashEqual(cmd, "actor"))
    {
        out->type = MapCmd_ACTOR;
    }
    else if (HashEqual(cmd, "kill"))
    {
        out->type = MapCmd_KILL;
    }
    else if (HashEqual(cmd, "take"))
    {
        out->type = MapCmd_TAKE;
    }
    else if (HashEqual(cmd, "take$"))
    {
        out->type = MapCmd_TAKE_MONEY;
    }
    else if (HashEqual(cmd, "give"))
    {
        out->type = MapCmd_GIVE;
    }
    else if (HashEqual(cmd, "walk"))
    {
        out->type = MapCmd_WALK;
    }
    else if (HashEqual(cmd, "flag"))
    {
        out->type = MapCmd_FLAG;
    }
    else if (HashEqual(cmd, "save"))
    {
        out->type = MapCmd_SAVE;
    }
    else if (HashEqual(cmd, "party"))
    {
        out->type = MapCmd_PARTY;
    }
    else if (HashEqual(cmd, "unparty"))
    {
        out->type = MapCmd_UNPARTY;
    }
    else if (HashEqual(cmd, "dead"))
    {
        out->type = MapCmd_DEAD;
    }
    else if (HashEqual(cmd, "block"))
    {
        out->type = MapCmd_BLOCK;
    }
    else if (HashEqual(cmd, "faction"))
    {
        out->type = MapCmd_FACTION;
    }
    else if (HashEqual(cmd, "hold"))
    {
        out->type = MapCmd_HOLD;
    }
    else if (HashEqual(cmd, "music"))
    {
        out->type = MapCmd_MUSIC;
    }
    else if (HashEqual(cmd, "model"))
    {
        out->type = MapCmd_MODEL;
    }
    else if (HashEqual(cmd, "texture"))
    {
        out->type = MapCmd_TEXTURE;
    }
    else if (HashEqual(cmd, "dir"))
    {
        out->type = MapCmd_DIR;
    }
    else if (HashEqual(cmd, "enter"))
    {
        out->type = MapCmd_ENTER;
    }
    else if (HashEqual(cmd, "hide"))
    {
        out->type = MapCmd_HIDE;
    }
}