#include "TMXLoader.h"
#include "TiledUtils.h"
#include <MapData.h>
#include <Hash.h>
#include <read/tinyxml2.h>
#include <MapEntityTypes.h>
#include <FileSystem.h>

using namespace tinyxml2;

bool TMXLoader::CanLoad(const char * path)
{
    XMLDocument doc;
    FileSystem::LoadXML(path, &doc);
    if (doc.Error())
    {
        Log_Error("Couldn't load map %s", path);
        return false;
    }
    
    XMLElement * map = doc.FirstChildElement("map");
    if (map == NULL)
    {
        Log_Error("Couldn't find valid map on %s", path);
        return false;
    }

    XMLElement * layer = map->FirstChildElement("layer");
    if (layer == NULL)
    {
        Log_Error("Couldn't find valid map layer on %s", path);
        return false;
    }

    XMLElement * data = layer->FirstChildElement("data");
    if (data == NULL)
    {
        Log_Error("Couldn't find valid map data on %s", path);
        return false;
    }
    
    u16 mapWidth = layer->IntAttribute("width");
    u16 mapDepth = layer->IntAttribute("height");
    if (mapWidth < 1 || mapDepth < 1 || mapWidth > 255 || mapDepth > 255)
    {
        Log_Error("Invalid map dimensions on %s (%dx%d)", path, mapWidth, mapDepth);
        return false;
    }
    return true;
}

Direction ParseDirStr(const char * s)
{
    if (s != NULL && s[0] != '\0')
    {
        char c = s[0];
        if (c >= 'a' && c <= 'z')
            c = c - 'a' + 'A';

        switch (c)
        {
            case 'N': return Direction_North;
            case 'E': return Direction_East;
            case 'S': return Direction_South;
            case 'W': return Direction_West;
        }
    }
    return Direction_None;
}

bool TMXLoader::Load(const char * path, MapData * mapData, u16 loadFlags)
{
    XMLDocument doc;
    FileSystem::LoadXML(path, &doc);
    if (doc.Error())
    {
        Log_Error("Couldn't load map %s", path);
        return false;
    }
    
    XMLElement * map = doc.FirstChildElement("map");
    if (map == NULL)
    {
        Log_Error("Couldn't find valid map on %s", path);
        return false;
    }

    XMLElement * layer = map->FirstChildElement("layer");
    if (layer == NULL)
    {
        Log_Error("Couldn't find valid map layer on %s", path);
        return false;
    }

    XMLElement * data = layer->FirstChildElement("data");
    if (data == NULL)
    {
        Log_Error("Couldn't find valid map data on %s", path);
        return false;
    }
    
    u16 mapWidth = layer->IntAttribute("width");
    u16 mapDepth = layer->IntAttribute("height");
    if (mapWidth < 1 || mapDepth < 1 || mapWidth > 255 || mapDepth > 255)
    {
        Log_Error("Invalid map dimensions on %s (%dx%d)", path, mapWidth, mapDepth);
        return false;
    }

    mapData->width = mapWidth;
    mapData->depth = mapDepth;
    mapData->zoom = 1.0f;

    strncpy(mapData->name, MapUtils::MapPathToName(path).c_str(), MAP_NAME_LENGTH);

    mapData->pixelsPerTile = map->UnsignedAttribute("tilewidth");

    XMLElement * properties = map->FirstChildElement("properties");
    if (properties != NULL)
    {
        XMLElement * property = properties->FirstChildElement();
        while (property != NULL)
        {
            const char * propName = property->Attribute("name");
            char layerName[16];
            for (i16 i=0; i<MAP_LAYER_COUNT; i++)
            {
                sprintf(layerName, "layer%d", i + 1);
                if (HashEqual(propName, layerName))
                {
                    mapData->layerHashes[i] = Hash(property->Attribute("value"));
                    break;
                }
            }
            if (HashEqual(propName, "water"))
            {
                mapData->layerHashes[MAP_LAYER_WATER] = Hash(property->Attribute("value"));
                break;
            }
            if (HashEqual(propName, "border"))
            {
                mapData->border = property->UnsignedAttribute("border");
            }
            if (HashEqual(propName, "env"))
            {
                mapData->envHash = Hash(property->Attribute("value"));
            }
            if (HashEqual(propName, "zoom"))
            {
                mapData->zoom = property->FloatAttribute("value", 1.0f);
            }
            if (HashEqual(propName, "music"))
            {
                const char * musicStr = property->Attribute("value");
                if (musicStr != NULL && musicStr[0] != '\0')
                {
                    mapData->music = MapPath(musicStr);
                }
            }
            property = property->NextSiblingElement();
        }
    }
    
    u16 tileCount = mapData->width * mapData->depth;
    mapData->tiles.resize(tileCount);

    const char * dataText = data->GetText();
    char buffer[16];
    u16 bufferI = 0;
    u16 tileI = 0;
    char c;
    bool writeTile;
    for (u32 i=0; i < UINT16_MAX; i++)
    {
        c = dataText[i];
        writeTile = false;
        if (c == '\0') 
        {
            writeTile = true;
        }
        else if (c >= '0' && c <= '9')
        {
            buffer[bufferI] = c;
            bufferI++;
        }
        else if (c == ',' && bufferI > 0)
        {
            writeTile = true;
        }

        if (writeTile)
        {
            buffer[bufferI] = '\0';
            TiledUtils::ExtractTileInfo(buffer, &mapData->tiles[tileI]);
            bufferI = 0;
            tileI++;
            if (c == '\0')
                break;
        }
    }

    XMLElement * objectGroup = map->FirstChildElement("objectgroup");
    if (objectGroup != NULL)
    {
        XMLElement * objectNode = objectGroup->FirstChildElement();
        properties = NULL;
        MapObject obj;
        while (objectNode != NULL)
        {
            obj.id = objectNode->IntAttribute("id");
            obj.cmdCount = 0;
            obj.flag = 0;
            obj.objType = MapObjectType_None;
            const char * type = objectNode->Attribute("type");
            if (type == NULL)
            {
                type = objectNode->Attribute("class");
            }
            if (type != NULL && type[0] != '\0')
            {
                if (HashEqual(type, "map"))
                {
                    obj.objType = MapObjectType_MapArea;
                }
                else if (HashEqual(type, "trigger"))
                {
                    obj.objType = MapObjectType_Trigger;
                }
                else if (HashEqual(type, "message"))
                {
                    obj.objType = MapObjectType_MessageArea;
                }
                else if (HashEqual(type, "load"))
                {
                    obj.objType = MapObjectType_Load;
                }
                else if (HashEqual(type, "spawn"))
                {
                    obj.objType = MapObjectType_SpawnPoint;
                }
                else if (HashEqual(type, "model"))
                {
                    obj.objType = MapObjectType_Entity;
                    obj.type = MapEntityTypes::ToId(MapEntityType_Model);
                }
            }
            
            if (obj.objType == MapObjectType_None)
            {
                u32 gid = objectNode->UnsignedAttribute("gid", UINT32_MAX);
                if (gid != UINT32_MAX)
                {
                    TiledUtils::ExtractGID(gid, &obj.type, &obj.flipH, &obj.flipV, &obj.flipD);
                    obj.objType = MapObjectType_Entity;
                }        
            }    

            obj.rotation = ((360 + objectNode->IntAttribute("rotation")) % 360) / 90;
            obj.x = objectNode->IntAttribute("x") / mapData->pixelsPerTile;
            obj.z = objectNode->IntAttribute("y") / mapData->pixelsPerTile;
            obj.width = objectNode->IntAttribute("width") / mapData->pixelsPerTile;
            obj.depth = objectNode->IntAttribute("height") / mapData->pixelsPerTile;
            obj.scale = obj.width < obj.depth ? obj.width : obj.depth;
            if (obj.scale == 0)
                obj.scale = 1;
            const char * objNameStr = objectNode->Attribute("name", NULL);
            if (objNameStr != NULL && objNameStr[0] != '#' && objNameStr[0] != '\n')
            {
                obj.nameHash = Hash(objNameStr);
                mapData->names.emplace_back(objNameStr);
                obj.name = mapData->names.back();
            }
            else
            {
                obj.name = NULL;
                obj.nameHash = 0;
            }

            properties = objectNode->FirstChildElement("properties");
            if (properties != NULL)
            {
                XMLElement * property = properties->FirstChildElement();
                while (property != NULL)
                {
                    const char * propName = property->Attribute("name");
                    const char * strValue = NULL;
                    MapCmd cmd;
                    if ((loadFlags & MapLoadFlag_NoCommands) == 0)
                    {
                        TiledUtils::ExtractMapCmd(propName, &cmd);
                    }
                    switch (cmd.type)
                    {
                        case MapCmd_IF:
                        case MapCmd_IFNOT:
                        case MapCmd_THEN:
                        case MapCmd_UNLOCK:
                        case MapCmd_GIVE:
                        case MapCmd_FLAG:
                        case MapCmd_PARTY:
                        case MapCmd_UNPARTY:
                        case MapCmd_KILL:
                        case MapCmd_TAKE:
                        case MapCmd_WALK:
                        case MapCmd_ACTOR:
                        case MapCmd_DEAD:
                        case MapCmd_FACTION:
                        case MapCmd_BLOCK:
                        case MapCmd_HOLD:
                        case MapCmd_HIDE:
                        case MapCmd_IF_MONEY:
                        case MapCmd_TAKE_MONEY:
                            cmd.value = (u8)property->UnsignedAttribute("value", 0);
                            break;

                        case MapCmd_DIR:
                            cmd.value = ParseDirStr(property->Attribute("value", NULL));
                            break;
                        
                        case MapCmd_MUSIC:
                        case MapCmd_SAVE:
                        case MapCmd_ENTER:
                            strValue = property->Attribute("value", NULL);
                            if (strValue == NULL)
                            {
                                Log_Error("No value on %s.%s", obj.name.Text(), propName);
                                cmd.type = MapCmd_NONE;
                            }
                            else
                            {
                                cmd.value = (u8)mapData->names.size();
                                mapData->names.emplace_back(strValue);
                            }
                            break;

                        case MapCmd_MAP:
                        case MapCmd_SAY:
                        case MapCmd_PROMPT:
                        case MapCmd_MODEL:
                        case MapCmd_TEXTURE:
                            strValue = property->Attribute("value", NULL);
                            if (strValue == NULL)
                            {
                                Log_Error("No value on %s.%s", obj.name.Text(), propName);
                                cmd.type = MapCmd_NONE;
                            }
                            else
                            {
                                cmd.value = (u8)mapData->messages.size();
                                mapData->messages.emplace_back(strValue);
                            }
                            break;
                    }
                    switch (cmd.type)
                    {
                        case MapCmd_NONE:
                            break;

                        case MapCmd_FLAG:
                            obj.flag = cmd.value;
                            break;

                        default:
                            if (obj.cmdCount == 0)
                            {
                                obj.cmdFirst = (u8)mapData->cmds.size();
                            }
                            mapData->cmds.emplace_back(cmd);
                            obj.cmdCount++;
                            break;
                    }
                    property = property->NextSiblingElement();
                }
            }

            if (obj.objType == MapObjectType_Entity)
            {
                obj.z -= 1;
                mapData->entities.emplace_back(obj);
            }
            else
            {
                mapData->areas.emplace_back(obj);
            }
            objectNode = objectNode->NextSiblingElement();
        }
    }

    return true;
}
