#include "FXParticles.h"
#include <Maths.h>
#include <Camera.h>
#include <RenderWindow.h>

FXParticles::FXParticles(u8 count, const char * texturePath) : 
    FXEntity(),
    count(count)
{
    bits = new ParticleBit[count]; 
    mesh.SetVertexCount(4);
    mesh.SetIndexCount(6);
    mesh.vertices[0] = Vector3(-0.5f, -0.5f, 0);
    mesh.vertices[1] = Vector3(0.5f, -0.5f, 0);
    mesh.vertices[2] = Vector3(0.5f, 0.5f, 0);
    mesh.vertices[3] = Vector3(-0.5f, 0.5f, 0);
    mesh.uvs[0] = Vector2(0, 0);
    mesh.uvs[1] = Vector2(1, 0);
    mesh.uvs[2] = Vector2(1, 1);
    mesh.uvs[3] = Vector2(0, 1);
    mesh.indices[0] = 0;
    mesh.indices[1] = 1;
    mesh.indices[2] = 3;
    mesh.indices[3] = 1;
    mesh.indices[4] = 2;
    mesh.indices[5] = 3;
    material = Material::Load(texturePath, "shaders/particle.dsh");
    mesh.material = material;
    lifeTime = 5.0f;
}

FXParticles::~FXParticles()
{
    delete bits;
    bits = NULL;
    Material::Free(material);
}

void FXParticles::Start()
{
    FXEntity::Start();
    float angleSep = PI * 2.0f / (float)count;
    for (u8 i=0; i<count; i++)
    {
        bits[i].scale = startSize;
        bits[i].position = Vector3_ZERO;
        bits[i].angle = Maths::RandF(PI * 2.0f);
        bits[i].velocity = Vector3(
            sinf(angleSep * i) * spread,
            0.1f * spread,
            cosf(angleSep * i) * spread
        );
        bits[i].spin = Maths::RandF(-spin, spin);
    }
}

void FXParticles::OnEnabled()
{
    FXEntity::OnEnabled();
    for (u8 i=0; i<count; i++)
    {
        bits[i].scale = startSize;
        bits[i].position = Vector3_ZERO;
        bits[i].angle = Maths::RandF(PI * 2.0f);
    }
}

void FXParticles::Update(float dt)
{
    FXEntity::Update(dt);
    ParticleBit swap;
    for (u8 i=0; i<count; i++)
    {
        bits[i].position += (bits[i].velocity + velocity) * dt;
        bits[i].angle += bits[i].spin * dt;
        bits[i].scale += grow * dt;

        for (u8 j = 1; j < count - i; j++)
        {
            if (bits[j].position.z < bits[i].position.z)
            {
                swap = bits[j];
                bits[j] = bits[i];
                bits[i] = swap;
            }
        }
    }
}

void FXParticles::Render()
{
    Camera * cam = Camera::Current();
    if (cam == NULL)
        return;

    RenderWindow * window = RenderWindow::Current();
    RenderMode prevRenderMode = window->GetRenderMode();

    window->SetRenderMode(renderMode);

    for (u8 i=0; i<count; i++)
    { 
        if (transform.position.y + bits[i].position.y < -1.0f)
            continue;
        Matrix mtx = glm::translate(Matrix(1.0f), transform.position + bits[i].position);
        Quat rot = glm::quatLookAt(glm::normalize(cam->transform.position - transform.position), Vector3_UP);
        rot *= Quat(Vector3(0, 0, bits[i].angle));
        mtx *= glm::mat4_cast(rot);
        mtx = glm::scale(mtx, Vector3(bits[i].scale));
        renderer.Render(&mesh, mtx);
    }
    
    window->SetRenderMode(prevRenderMode);
}