#ifndef _TARGETING_BEHAVIOUR_H
#define _TARGETING_BEHAVIOUR_H

#include <ActorBehaviour.h>
#include <Scene.h>
#include <Maths.h>
#include <AttackManager.h>

enum TargetingState
{
    TargetingState_Search,
    TargetingState_Attack,
    TargetingState_Chill,
    TargetingState_Return,
    TargetingState_COUNT
};

class TargetingBehaviour : public ActorBehaviour
{
private:
    ActorBehaviour * behaviours[TargetingState_COUNT];
    float chillTimer = 0.0f;
    float searchTimer = 0.0f;
    float chillDuration = 1.0f;
    float randomness = 0.3f;
    float attackDowntime = 0.5f;

public:
    float detectRange;
    float searchDuration;
    float maxDistFromOrigin;
    float attackOutsideOriginDuration = 4.0f;

    TargetingBehaviour(
        Actor * actor, 
        ActorBehaviour * bSearch,
        ActorBehaviour * bAttack,
        float detectRange = 6.0f,
        float searchDuration = 2.0f,
        float maxDistFromOrigin = 10.0f
    ) :
        ActorBehaviour(actor),
        detectRange(detectRange),
        searchDuration(searchDuration),
        maxDistFromOrigin(maxDistFromOrigin)
    {
        behaviours[TargetingState_Search] = bSearch;
        behaviours[TargetingState_Attack] = bAttack;
        behaviours[TargetingState_Chill] = NULL;
        behaviours[TargetingState_Return] = NULL;
    }

    virtual ~TargetingBehaviour()
    {
        AttackManager::Remove(actor);
        delete behaviours[TargetingState_Search];
        delete behaviours[TargetingState_Attack];   
    }

    void Start() 
    {
        target = NULL;
        ActorBehaviour::Start();
        AttackManager::Remove(actor);
    }

    u8 UpdateState(float dt)
    {
        switch (state)
        {
            case TargetingState_Attack:
                return UpdateAttack(dt);
            case TargetingState_Chill:
                return UpdateChill(dt);
            case TargetingState_Return:
                return UpdateReturn(dt);
        }
        return UpdateSearch(dt);
    }

    TargetingState StartSearch()
    {
        AttackManager::Remove(actor);
        target = actor->FindTarget();
        behaviours[TargetingState_Search]->SetTarget(target);
        if (behaviours[TargetingState_Search]->IsRunning() == false)
        {
            behaviours[TargetingState_Search]->Start();
        }
        searchTimer = Maths::RandFMultiplier(searchDuration, randomness);
        return TargetingState_Search;
    }

    TargetingState UpdateSearch(float dt)
    {
        if (Maths::IsWithinDistance(actor->transform.position, actor->GetSpawnPos(), maxDistFromOrigin) == false)
        {
            return TargetingState_Return;
        }
        if (attackDowntime > 0.0f)
        {
            attackDowntime -= dt;
        }
        behaviours[TargetingState_Search]->Update(dt);
        if (attackDowntime <= 0.0f && Entity::IsActive(target) && actor->IsWithinRange(target, detectRange, minDot))
        {
            if (AttackManager::GetAttackerCount() == 0)
            {
                return StartAttack();
            }
        }
        searchTimer -= dt;
        if (searchTimer <= 0.0f)
        {
            return StartSearch();
        }
        return TargetingState_Search;
    }

    TargetingState StartAttack()
    {
        behaviours[TargetingState_Attack]->SetTarget(target);
        behaviours[TargetingState_Attack]->Start();
        AttackManager::Add(actor);
        return TargetingState_Attack;
    }

    TargetingState UpdateAttack(float dt)
    {
        if (stateTime > attackOutsideOriginDuration && Maths::IsWithinDistance(actor->transform.position, actor->GetSpawnPos(), maxDistFromOrigin) == false)
        {
            return TargetingState_Return;
        }
        behaviours[TargetingState_Attack]->Update(dt);
        if (target == NULL || behaviours[TargetingState_Attack]->IsDone())
        {
            behaviours[TargetingState_Attack]->Start();
            return StartSearch();
        }
        return TargetingState_Attack;
    }

    TargetingState StartChill()
    {
        behaviours[TargetingState_Search]->SetTarget(target);
        behaviours[TargetingState_Search]->Start();
        chillTimer = Maths::RandFMultiplier(chillDuration, randomness);
        return TargetingState_Chill;
    }

    TargetingState UpdateChill(float dt)
    {
        chillTimer -= dt;
        if (chillTimer <= 0.0f)
        {
            return StartSearch();
        }
        return TargetingState_Chill;
    }

    TargetingState UpdateReturn(float dt)
    {
        targetPos = actor->GetSpawnPos();
        if (Maths::IsWithinDistance(actor->transform.position, targetPos, 2.0f))
        {
            return StartSearch();
        }
        return TargetingState_Return;
    }
    
    void Chill()
    {
        target = NULL;
        targetPos = actor->transform.position;
        state = StartChill();
    }

    Vector3 GetMoveVelocity(float dt)
    {
        if (state < TargetingState_Chill)
        {
            return behaviours[state]->GetMoveVelocity(dt);
        }
        return Vector3_ZERO;
    }
};

#endif