#ifndef _INTERACT_BEHAVIOUR_H
#define _INTERACT_BEHAVIOUR_H

#include <ActorBehaviour.h>
#include <Actor.h>
#include <ui/GameUI.h>

enum InteractState
{
    InteractState_Chase = 1,
    InteractState_Windup,
    InteractState_Act,
    InteractState_Done
};

#define INTERACT_DISTANCE_DEFAULT 1.4f
#define INTERACT_WINDUP_DURATION_DEFAULT 0.1f

class InteractBehaviour : public ActorBehaviour
{
protected:
    float interactRange = INTERACT_DISTANCE_DEFAULT;
    float interactWindupDuration = INTERACT_WINDUP_DURATION_DEFAULT;

public:
    InteractBehaviour(Actor * actor) : 
        ActorBehaviour(actor)
    {
    }

    virtual void Start()
    {
        if (target == NULL)
        {
            target = actor->FindTarget();
        }
        if (target == NULL)
        {
            Done();
            return;
        }
        ActorBehaviour::Start();
    }

    virtual u8 PickState()
    {
        return InteractState_Chase;
    }

    virtual u8 UpdateState(float dt)
    {
        switch (state)
        {
            case InteractState_Chase:
                return UpdateChase(dt);

            case InteractState_Windup:
                return UpdateWindup(dt);

            case InteractState_Act:
                return UpdateAct(dt);

            case InteractState_Done:
                return UpdateDone(dt);
        }
        return InteractState_Chase;
    }

    virtual InteractState UpdateChase(float dt)
    {
        if (target == NULL)
            return InteractState_Done;

        targetPos = target->transform.position;
        if (actor->IsWithinRange(targetPos, interactRange))
        {
            return BeginWindup();
        }
        return InteractState_Chase;
    }

    virtual InteractState BeginWindup()
    {
        return InteractState_Windup;
    }

    virtual InteractState UpdateWindup(float dt)
    {
        if (stateTime > interactWindupDuration)
        {
            return BeginAct();
        }
        targetPos = actor->transform.position;
        return InteractState_Windup;
    }

    virtual InteractState BeginAct() = 0;

    virtual InteractState UpdateAct(float dt) = 0;

    InteractState UpdateDone(float dt)
    {
        target = NULL;
        Done();
        return InteractState_Done;
    }
};

#endif