#ifndef _TILE_MAP_H
#define _TILE_MAP_H

#include <desstypes.h>
#include <Entity.h>
#include <MeshRenderer.h>
#include <EntityFactory.h>
#include <MapData.h>
#include <MapEntityTypes.h>
#include <Mesh.h>

const u8 MAP_MESH_COUNT = MAP_LAYER_COUNT * 2;

class Material;
class MapEntity;
struct MapLayerInfo;

class TileMapLoadListener
{
public:
    virtual void BeforeMapLoad() {}
    virtual void OnMapLoaded(const char *mapName) = 0;
};

enum TileMapLoadState
{
    TileMapLoadState_INIT,
    TileMapLoadState_LOADED_FILE,
    TileMapLoadState_UNLOADING,
    TileMapLoadState_ENVIRONMENT,
    TileMapLoadState_BUILD_TILES,
    TileMapLoadState_BUILD_EDGES,
    TileMapLoadState_BUILD_ENTITIES,
    TileMapLoadState_FIRE_TRIGGERS,
    TileMapLoadState_COMPLETE,
    TileMapLoadState_ABORTED
};

struct TileMapLoadInfo
{
    TileMapLoadState state = TileMapLoadState_INIT;
    u16 loadFlags = 0;
    u16 index = 0;
    u16 count = 0;
    u16 entitiesPerFrame = 0;
    TileMapLoadListener * listener = NULL;
    MapData data;
    char path[MAP_PATH_LENGTH];
};

class TileMap : public Entity
{
public:
    MapData data;
    u8 flags[MAP_FLAG_COUNT];
    bool allDead = false;

    static bool IsCornerType(TileType type);
    static Vector3 GetCornerForward(TileType type);
    static bool IsRampType(TileType type);
static Direction GetRampDir(TileType type);
    static Vector3 GetRampForward(TileType type);
    static TileType FlipTypeDir(TileType type);

    TileMap();
    virtual ~TileMap();

    void Added();

    void Unload();
    TileMapLoadInfo * Load(const char * path, u16 loadFlags = 0, TileMapLoadListener * listener = NULL);
    bool LoadContinue();
    void SetLayerMaterial(u8 layer, LayerType layerType, Ref<Material> & material);
    void SetLayerMaterial(u8 layer, LayerType layerType, const char * texturePath, const char * shaderPath = NULL);
    void ClearLayerMaterials();

    float GetHeight(Vector3 position) const;
    float GetHeight(i16 x, i16 z, float cx = 0.5f, float cz = 0.5f) const;
    bool IsInside(Vector3 position) const;
    const MapObject * GetArea(Vector3 position) const;
    const MapObject * GetAreaByName(const char * areaName) const;
    const MapObject * GetAreaByHash(HashInt areaNameHash) const;
    const MapObject * GetArea(u16 id) const;
    const MapCmd * GetObjectCmd(const MapObject * obj, MapCmdType type) const;
    const MapObject * GetEntityById(u16 id) const;
    const MapObject * GetObjectById(u16 id) const;
    const MapObject * GetEntityByName(const char * entityName) const;
    const MapObject * GetEntityByHash(HashInt entityNameHash) const;
    const MapObject * GetEntityInfo(MapEntityType type) const;
    Vector3 GetPosition(i16 x, i16 z, float cx = 0.5f, float cz = 0.5f) const;
    Vector3 ClampPosition(Vector3 p, float pad = 0.1f) const;
    void ClampOrds(i16 * x, i16 * z) const;
    bool GetTile(Vector3 position, i16 * outX, i16 * outZ) const;
    const TileInfo & GetTileInfo(i16 x, i16 z) const;
    const TileInfo & GetTileInfo(Vector3 position) const;
    const MapLayerInfo * GetLayerInfo(u8 layer);
    const MapLayerInfo * GetLayerInfo(i16 x, i16 z);
    float GetLayerHeight(u8 layer) const;

    void UpdateFlags();

    void Render();

    void SetEntityFactory(EntityFactory *value);
    
    void FlagSet(u8 flag, bool on);
    bool FlagIsSet(u8 flag);
    void FlagsClear();

    void GetAreaEdge(const MapObject * area, Vector3 direction, i16 & out_x, i16 & out_z) const;

    inline bool IsLoading() const { return loadInfo.state > TileMapLoadState_INIT && loadInfo.state < TileMapLoadState_COMPLETE; }

private:
    TileMapLoadInfo loadInfo;
    EntityFactory * entityFactory = NULL;
    Mesh meshes[MAP_MESH_COUNT];
    Ref<Material> layerMaterials[MAP_MESH_COUNT];
    MeshRenderer meshRenderer;
    
    TileMapLoadState LoadNextBit();
    bool LoadMapFile(const char * path);
    bool LoadUnload();
    bool LoadEnvironment();
    bool LoadBuildTiles();
    bool LoadBuildEdges();
    bool LoadEntities();
    bool LoadTriggers();

    bool BuildTile(const TileInfo &info, i16 x, i16 z);
    void BuildSurface(Mesh *mesh, TileType type, i16 x, i16 z, u8 layer);
    void BuildWall(Mesh *mesh, TileType type, i16 x, i16 z, Direction direction, u8 layerMax, u8 layerMin);
    void BuildWallCorner(Mesh *mesh, TileType type, i16 x, i16 z, u8 layerMax, u8 layerMin);
    void BuildRamp(const TileInfo &info, Direction direction, i16 x, i16 z);
    void BuildQuad(Mesh *mesh, Vector3 a, Vector3 b, Vector3 c, Vector3 d, Vector3 normal, Vector2 maxUv = Vector2(1, 1));
    void BuildEdges();

    Mesh *GetMesh(u8 layer, LayerType layerType);

    void UpdateLayerMaterials();
};

#endif