#include <TileMap.h>
#include <desstypes.h>
#include <vector>
#include <Mesh.h>
#include <read/tinyxml2.h>
#include <glm/vec3.hpp>
#include <Scene.h>
#include <MapEntity.h>
#include <Hash.h>
#include <Maths.h>
#include <MapEnvironment.h>
#include <FileSystem.h>
#include <loaders/TMXLoader.h>
#include <Save.h>
#include <MapEntity.h>
#include <MapEntityTypes.h>
#include <Sequencer.h>
#include <MapSequence.h>
#include <MusicPlayer.h>

using namespace tinyxml2;

const float OVERFLOW_XZ = 0.01f;
const float HYP_1 = 1.4142f;
const float HYP_0 = HYP_1 * 0.5;
const float HYP_I = 1.0f / HYP_1;
const float GROUND_BOTTOM = -1.0f;

const TileInfo TILE_INFO_EMPTY (TileType_Empty, 0, 0);

TileMap::TileMap()
{
    for (u8 i = 0; i < MAP_FLAG_COUNT; i++)
    {
        flags[i] = 0;
    }
}

TileMap::~TileMap()
{   
    Unload();
}

void TileMap::Added()
{
    for (std::vector<MapObject>::const_iterator it = data.entities.begin(); it != data.entities.end(); ++it)
    {
        if (Entity::NotDestroyed(it->entity) && it->entity->scene != scene)
        {
            scene->AddEntity(it->entity);
        }
    }
}

bool TileMap::IsCornerType(TileType type)
{
    return type >= TileType_CornerNE && type <= TileType_CornerNW;
}

bool TileMap::IsRampType(TileType type)
{
    return type >= TileType_RampN && type <= TileType_RampW;
}

TileType TileMap::FlipTypeDir(TileType type)
{
	switch (type)
    {
		case TileType_CornerNE:
			return TileType_CornerSW;
		case TileType_CornerNW:
			return TileType_CornerSE;
		case TileType_CornerSE:
			return TileType_CornerNW;
		case TileType_CornerSW:
			return TileType_CornerNE;

        case TileType_RampN:
            return TileType_RampS;
        case TileType_RampE:
            return TileType_RampW;
        case TileType_RampS:
            return TileType_RampN;
        case TileType_RampW:
            return TileType_RampE;
    }
	return type;
}

bool IsCornerDir(TileType type, Direction dir)
{
	switch (type)
    {
		case TileType_CornerNE:
			return Directions::GetZ(dir) >= 0 && Directions::GetX(dir) <= 0;
		case TileType_CornerNW:
			return Directions::GetZ(dir) >= 0 && Directions::GetX(dir) >= 0;
		case TileType_CornerSE:
			return Directions::GetZ(dir) <= 0 && Directions::GetX(dir) <= 0;
		case TileType_CornerSW:
			return Directions::GetZ(dir) <= 0 && Directions::GetX(dir) >= 0;
    }
	return false;
}


const TileInfo & TileMap::GetTileInfo(i16 x, i16 z) const
{
    ClampOrds(&x, &z);
    if (x < data.width && z < data.depth)
    {
        size_t index = x + z * data.width;
        if (index >= 0 && index < data.tiles.size())
            return data.tiles[index];
    }
    return TILE_INFO_EMPTY;
}


const TileInfo & TileMap::GetTileInfo(Vector3 position) const
{
    i16 x, z;
    GetTile(position, &x, &z);
    return GetTileInfo(x, z);
}

bool TileMap::IsInside(Vector3 position) const
{
    i16 x;
    return GetTile(position, &x, &x);
}

float TileMap::GetHeight(Vector3 position) const
{
    i16 x, z;
    GetTile(position, &x, &z);
    float cx = fmod(position.x - data.tileSize * 0.5f, data.tileSize) / data.tileSize;
    float cz = fmod(position.z - data.tileSize * 0.5f, data.tileSize) / data.tileSize;
    return GetHeight(x, z, cx, cz);
}

float TileMap::GetHeight(i16 x, i16 z, float cx, float cz) const
{
    ClampOrds(&x, &z);
    const TileInfo & info = GetTileInfo(x, z);

    switch (info.type)
    {
        // RAMPS
        case TileType_RampN:
            return (1.0f - cz) * GetLayerHeight(info.layer) + cz * GetLayerHeight(info.under);
        case TileType_RampS:
            return (1.0f - cz) * GetLayerHeight(info.under) + cz * GetLayerHeight(info.layer);
        case TileType_RampE:
            return (1.0f - cx) * GetLayerHeight(info.under) + cx * GetLayerHeight(info.layer);
        case TileType_RampW:
            return (1.0f - cx) * GetLayerHeight(info.layer) + cx * GetLayerHeight(info.under);

        // CORNERS
        case TileType_CornerNW:
            return cx + cz > 1.0f ? GetLayerHeight(info.under) : GetLayerHeight(info.layer);
        case TileType_CornerNE:
            return (1.0f - cx) + cz > 1.0f ? GetLayerHeight(info.under) : GetLayerHeight(info.layer);
        case TileType_CornerSW:
            return (1.0f - cx) + cz > 1.0f ? GetLayerHeight(info.layer) : GetLayerHeight(info.under);
        case TileType_CornerSE:
            return cx + cz > 1.0f ? GetLayerHeight(info.layer) : GetLayerHeight(info.under);
    }
    return GetLayerHeight(info.layer);
}

float TileMap::GetLayerHeight(u8 layer) const 
{
    if (layer == 0)
        return GROUND_BOTTOM;
    if (layer == MAP_LAYER_WATER + 1)
        return MAP_WATER_HEIGHT;
    return ((float)layer - 1.0f) * data.tileHeight;
}

const MapLayerInfo * TileMap::GetLayerInfo(u8 layer)
{
    if (layer == 0)
        return NULL;
        
    layer = (layer - 1) % MAP_LAYER_COUNT;  
    
    HashInt hash = data.layerHashes[layer];
    if (hash > 0)
    {
        return MapEnvironment::GetLayerInfo(hash);
    }
    
    const MapEnvInfo * envInfo = MapEnvironment::GetEnvInfo(data.envHash);
    if (envInfo != NULL)
    {
        return envInfo->layers[layer];
    }

    return NULL;
}

const MapLayerInfo * TileMap::GetLayerInfo(i16 x, i16 z)
{
    return GetLayerInfo(GetTileInfo(x, z).layer);
}

const MapObject * TileMap::GetArea(Vector3 position) const
{
    i16 x, z;
    GetTile(position, &x, &z);
    for (std::vector<MapObject>::const_iterator it = data.areas.begin(); it != data.areas.end(); ++it)
    {
        if (it->Contains(x, z))
            return &(*it);
    }
    return NULL;
}

const MapCmd * TileMap::GetObjectCmd(const MapObject * obj, MapCmdType type) const
{
    if (obj != NULL && type != MapCmd_NONE && obj->cmdCount > 0)
    {
        const MapCmd * cmd = NULL;
        for (u8 i=0; i<obj->cmdCount; i++)
        {
            cmd = &data.cmds[i + obj->cmdFirst];
            if (cmd != NULL && cmd->type == type)
                return cmd;
        }
    }
    return NULL;
}

const MapObject * TileMap::GetAreaByName(const char * areaName) const
{
    return GetAreaByHash(Hash(areaName));
}

const MapObject * TileMap::GetAreaByHash(HashInt areaNameHash) const
{
    if (areaNameHash > 0)
    {
        for (std::vector<MapObject>::const_iterator it = data.areas.begin(); it != data.areas.end(); ++it)
        {
            if (it->nameHash == areaNameHash)
                return &(*it);
        }
    }
    return NULL;
}

const MapObject * TileMap::GetArea(u16 id) const
{
    for (std::vector<MapObject>::const_iterator it = data.areas.begin(); it != data.areas.end(); ++it)
    {
        if (it->id == id)
            return &(*it);
    }
    return NULL;
}

const MapObject * TileMap::GetEntityById(u16 id) const
{
    for (std::vector<MapObject>::const_iterator it = data.entities.begin(); it != data.entities.end(); ++it)
    {
        if (it->id == id)
            return &(*it);
    }
    return NULL;
}

const MapObject * TileMap::GetObjectById(u16 id) const
{
    const MapObject * result = GetEntityById(id);
    if (result != NULL)
        return result;

    return GetArea(id);
}

const MapObject * TileMap::GetEntityByName(const char * entityName) const
{
    return GetEntityByHash(Hash(entityName));
}

const MapObject * TileMap::GetEntityByHash(HashInt entityNameHash) const
{
    if (entityNameHash > 0)
    {
        for (std::vector<MapObject>::const_iterator it = data.entities.begin(); it != data.entities.end(); ++it)
        {
            if (it->nameHash == entityNameHash)
                return &(*it);
        }
    }
    return NULL;
}

const MapObject * TileMap::GetEntityInfo(MapEntityType type) const
{
    for (std::vector<MapObject>::const_iterator it = data.entities.begin(); it != data.entities.end(); ++it)
    {
        if (MapEntityTypes::ToType(it->type) == type)
            return &(*it);
    }
    return NULL;
}


void TileMap::FlagSet(u8 flag, bool on)
{
    if (flag == 0)
        return;

    bool isSet = FlagIsSet(flag);
    if (on == isSet)
        return;

    if (on)
    {
        for (u8 i = 0; i < MAP_FLAG_COUNT; i++)
        {
            if (flags[i] == 0)
            {
                flags[i] = flag;
                return;
            }
        }
    }
    else
    {
        for (u8 i = 0; i < MAP_FLAG_COUNT; i++)
        {
            if (flags[i] == flag)
            {
                flags[i] = 0;
            }
        }
    }
}

bool TileMap::FlagIsSet(u8 flag)
{
    if (flag > 0)
    {
        for (u8 i = 0; i < MAP_FLAG_COUNT; i++)
        {
            if (flags[i] == flag)
                return true;
        }
    }
    return false;
}

void TileMap::FlagsClear()
{
    Log_Info("Clearing flags...\n");
    for (u8 i = 0; i < MAP_FLAG_COUNT; i++)
    {
        flags[i] = 0;
    }
    Log_Info("Cleared flags...\n");
}

Vector3 TileMap::GetPosition(i16 x, i16 z, float cx, float cz) const
{
    Vector3 result(
        (float)x * data.tileSize + (Maths::Clamp01(cx) - 0.5f) * data.tileSize - OVERFLOW_XZ,
        GetHeight(x, z, cx, cz),
        (float)z * data.tileSize + (Maths::Clamp01(cz) - 0.5f) * data.tileSize - OVERFLOW_XZ
    );
    return result;
}

void TileMap::GetAreaEdge(const MapObject * area, Vector3 direction, i16 & out_x, i16 & out_z) const
{
    Vector3 cardinal = Maths::Cardinal(direction);
    out_x = (i16)((float)area->x + ((float)area->width * (cardinal.x + 1.0f)) * 0.5f);
    out_z = (i16)((float)area->z + ((float)area->depth * (cardinal.z + 1.0f)) * 0.5f);
}

float ClampPos(float value, float min, float max)
{
    if (min > max)
        return (min + max) * 0.5f;
    return value < min ? min : value > max ? max : value;
}

Vector3 TileMap::ClampPosition(Vector3 p, float pad) const
{
    float offset = data.tileSize * - 0.5f;
    p.x = ClampPos(p.x, offset + pad, (data.width) * data.tileSize + offset - pad);
    p.z = ClampPos(p.z, offset + pad, (data.depth) * data.tileSize + offset - pad);
    p.y = GetHeight(p);
    return p;
}

void TileMap::ClampOrds(i16 * x, i16 * z) const
{
    if (*x < 0)
        *x = 0;
    if (*z < 0)
        *z = 0;
    if (*x > data.width - 1)
        *x = data.width - 1;
    if (*z > data.depth - 1)
        *z = data.depth - 1;
}

bool TileMap::GetTile(Vector3 position, i16 * outX, i16 * outZ) const
{
    ClampOrds(outX, outZ);
    float fx = position.x / data.tileSize + 0.5f;
    float fz = position.z / data.tileSize + 0.5f;
    *outX = (i16)fx;
    *outZ = (i16)fz;
    if (*outX < 0 || *outZ < 0 || *outX > data.width - 1 || *outZ > data.depth - 1)
        return false;
    return true;
}

Direction TileMap::GetRampDir(TileType type) 
{
    switch (type)
    {
        case TileType_RampN:
            return Direction_North;
        case TileType_RampS:
            return Direction_South;
        case TileType_RampE:
            return Direction_East;
        case TileType_RampW:
            return Direction_West;
    }
    return Direction_None;
}

Vector3 TileMap::GetCornerForward(TileType type)
{
    switch (type)
    {
        case TileType_CornerNW:
            return Vector3(-HYP_I, 0, -HYP_I);

        case TileType_CornerNE:
            return Vector3(HYP_I, 0, -HYP_I);

        case TileType_CornerSE:
            return Vector3(HYP_I, 0, HYP_I);

        case TileType_CornerSW:
            return Vector3(-HYP_I, 0, HYP_I);
    }
    return Vector3_ZERO;
}

Vector3 TileMap::GetRampForward(TileType type)
{
    Direction dir = GetRampDir(type);
    return Directions::GetForward(dir);
}

void TileMap::Render()
{
    Matrix matrix = transform.GetMatrix();
    for (u8 i=0; i<MAP_MESH_COUNT; i++)
    {
        meshRenderer.Render(&meshes[i], matrix);
    }
}

void TileMap::SetEntityFactory(EntityFactory * value)
{
    entityFactory = value;
}

TileMapLoadInfo * TileMap::Load(const char * path, u16 loadFlags, TileMapLoadListener * listener)
{
    loadInfo = TileMapLoadInfo();
    loadInfo.loadFlags = loadFlags;
    loadInfo.listener = listener;
    strncpy(loadInfo.path, path, MAP_PATH_LENGTH);
    loadInfo.state = LoadMapFile(path) ? TileMapLoadState_LOADED_FILE : TileMapLoadState_ABORTED;
    LoadContinue();
    return &loadInfo;
}

bool TileMap::LoadContinue()
{
    TileMapLoadState nextState = LoadNextBit();
    if (nextState != loadInfo.state)
    {
        if (nextState == TileMapLoadState_ABORTED)
        {
            Log_Error("Tile map aborted on phase %i\n", loadInfo.state);
        }
        else if (nextState == TileMapLoadState_COMPLETE)
        {
            Log_Info("Tile map loaded.\n");
        }
        loadInfo.state = nextState;
        loadInfo.index = 0;
        loadInfo.count = 0;
    }
    return nextState < TileMapLoadState_COMPLETE;
}

TileMapLoadState TileMap::LoadNextBit()
{
    switch (loadInfo.state)
    {
        case TileMapLoadState_LOADED_FILE:
        case TileMapLoadState_UNLOADING:
            return LoadUnload() ? TileMapLoadState_ENVIRONMENT : TileMapLoadState_ABORTED;

        case TileMapLoadState_ENVIRONMENT:
            if (!LoadEnvironment())
                return TileMapLoadState_ABORTED;
            if (!LoadBuildTiles())
                return TileMapLoadState_BUILD_TILES;
            return LoadBuildEdges() ? TileMapLoadState_BUILD_ENTITIES : TileMapLoadState_BUILD_EDGES;

        case TileMapLoadState_BUILD_ENTITIES:
            return LoadEntities() ? TileMapLoadState_FIRE_TRIGGERS : TileMapLoadState_BUILD_ENTITIES;

        case TileMapLoadState_FIRE_TRIGGERS:
            return LoadTriggers() ? TileMapLoadState_COMPLETE : TileMapLoadState_ABORTED;

        case TileMapLoadState_COMPLETE:
            return TileMapLoadState_COMPLETE;
    }
    return TileMapLoadState_ABORTED;
}

bool TileMap::LoadMapFile(const char * path)
{
    Log_Info("\nLoad map requested %s...\n", path);
    MapLoader * loader = NULL;
    if (FileSystem::IsExtenstion(path, "tmx"))
    {
        loader = new TMXLoader();
        if (loader->CanLoad(path) == false)
        {
            Log_Error("Map load unsupported: %s", path);
            return false;
        }
    }
    else
    {
        Log_Error("Unsupported file format: %s", path);
        return false;
    }
    
    Log_Info("Loading map data...\n");
    loadInfo.data = MapData();
    if (loader->Load(path, &loadInfo.data, loadInfo.loadFlags) == false)
    {
        Log_Error("Map load aborted: %s", path);
        return false;
    }

    return true;
}

void TileMap::Unload()
{
    Log_Info("Clearing meshes...\n");
    for (u8 i = 0; i < MAP_MESH_COUNT; i++)
    {
        meshes[i].Clear();
    }
    
    Log_Info("Clearing entities...\n");
    if (data.entities.size() > 0)
    {
        for (std::vector<MapObject>::const_iterator it = data.entities.begin(); it != data.entities.end(); ++it)
        {
            if (scene->HasEntity(it->entity))
            {
                it->entity->Destroy();
            }
        }
    }
    data.entities.clear();
    data.Clear();
    FlagsClear();
    allDead = false;
}
    
bool TileMap::LoadUnload()
{
    if (loadInfo.listener != NULL)
    {
        loadInfo.listener->BeforeMapLoad();
    }
    
    Unload();

    return true;
}

bool TileMap::LoadEnvironment()
{
    Log_Info("Loading environment...\n");

    data = loadInfo.data;
    if (data.envHash > 0)
    {
        const MapEnvInfo * envInfo = MapEnvironment::GetEnvInfo(data.envHash);
        if (envInfo != NULL)
        {
            for (u8 i = 0; i < MAP_LAYER_COUNT; i++)
            {
                if (envInfo->layers[i] == NULL)
                    continue;
                SetLayerMaterial(i, LayerType_Ground, MapStr::ReadText(&envInfo->layers[i]->textureSurface), MapStr::ReadText(&envInfo->layers[i]->shader));
                SetLayerMaterial(i, LayerType_Wall, MapStr::ReadText(&envInfo->layers[i]->textureEdge), MapStr::ReadText(&envInfo->layers[i]->shader));
            }
            if (data.border == UINT8_MAX)
            {
                data.border = envInfo->border;
            }
        }
    }

    if (data.border == UINT8_MAX)
    {
        data.border = DEFAULT_BORDER;
    }

    Log_Info("Loading additional layers...\n");
    for (u8 i = 0; i < MAP_LAYER_COUNT; i++)
    {
        u32 layerHash = data.layerHashes[i];
        if (layerHash > 0)
        {
            const MapLayerInfo * layerInfo = MapEnvironment::GetLayerInfo(layerHash);
            if (layerInfo != NULL)
            {
                SetLayerMaterial(i, LayerType_Ground, MapStr::ReadText(&layerInfo->textureSurface));
                SetLayerMaterial(i, LayerType_Wall, MapStr::ReadText(&layerInfo->textureEdge));
            }
        }
    }

    return true;
}

bool TileMap::LoadBuildTiles()
{
    if (loadInfo.index == 0)
    {
        loadInfo.count = data.width * data.depth;
    }
    u16 end = loadInfo.index + 10;
    //if (end > loadInfo.count)
    {
        end = loadInfo.count;
    }
    Log_Info("Building tiles %i to %i...\n", loadInfo.index, end);
    do
    {
        if (BuildTile(data.tiles[loadInfo.index], loadInfo.index % data.width, loadInfo.index / data.width) == false)
        {
            return false;
        }
    } 
    while (++loadInfo.index < end);
    if (end == loadInfo.count)
    {
        UpdateLayerMaterials();
        Log_Info("Tiles built\n");
        return true;
    }
    return false;
}

bool TileMap::LoadBuildEdges()
{
    if (data.border > 0)
    {
        Log_Info("Building border...\n");
        BuildEdges();
    }
    return true;
}

bool TileMap::LoadEntities()
{   
    Log_Info("Loading entities...\n");
    if (entityFactory == NULL)
    {
        Log_Error("No entity factory specified.\n");
        return true;
    }

    size_t entityCount = data.entities.size();
    if (entityCount < 1)
    {
        return true;
    }

    size_t limit = entityCount;
    if (loadInfo.entitiesPerFrame > 0)
    {
        limit = loadInfo.index + loadInfo.entitiesPerFrame;
    }
    if (limit > entityCount)
    {
        limit = entityCount;
    }

    MapObject * obj = NULL;
    for (; loadInfo.index < limit; loadInfo.index++)
    {
        obj = &data.entities[loadInfo.index];
        MapEntityType mapEntityType = MapEntityTypes::ToType(obj->type);
        if (((loadInfo.loadFlags & MapLoadFlag_NoActors) != 0) && MapEntityTypes::IsActor(mapEntityType))
            continue;

        if (entityFactory->CanCreate(obj->type))
        {
            Vector3 position = GetPosition(obj->x, obj->z);
            obj->entity = entityFactory->CreateId(obj->type);
            if (obj->entity != NULL)
            {
                if (obj->entity->renderQueue == 0)
                {
                    obj->entity->renderQueue = renderQueue;
                }
                if (obj->rotation == 3 || obj->rotation == 2)
                {
                    position.x -= data.tileSize * obj->scale;
                }
                if (obj->rotation == 1 || obj->rotation == 2)
                {
                    position.z += data.tileSize * obj->scale;
                }
                Vector3 scale((float)obj->scale);
                if (obj->flipH) 
                {
                    scale.x = -scale.x;
                }
                if (obj->flipV) 
                {
                    scale.z = -scale.z;
                }
                if (obj->scale > 1)
                {
                    position.x += data.tileSize * 0.5f * (float)(obj->scale - 1);
                    position.z -= data.tileSize * 0.5f * (float)(obj->scale - 1);
                }
                position.y = GetHeight(position);
                obj->entity->transform.position = position;
                obj->entity->transform.rotation = glm::angleAxis(PI * -0.5f * (float)(obj->rotation), Vector3_UP);
                obj->entity->transform.scale = scale;
                obj->entity->name = MapStr::ReadText(&obj->name);
                if (scene != NULL)
                {
                    scene->AddEntity(obj->entity);
                }

                MapEntity* me = (MapEntity*)(obj->entity);
                if (me != NULL)
                {
                    me->map = this;
                    me->info = obj;
                    me->entityOwner = this;
                }
            }
        }
        else
        {
            obj->entity = NULL;
        }
    }
    Log_Info("Loaded entities\n");
    return loadInfo.index >= entityCount - 1;
}

bool TileMap::LoadTriggers()
{
    Log_Info("Notifying load listener...\n");
    if (loadInfo.listener != NULL)
    {
        loadInfo.listener->OnMapLoaded(loadInfo.path);
    }

    // sequence on load
    if ((loadInfo.loadFlags & MapLoadFlag_NoCommands) == 0)
    {
        Log_Info("Loading sequences...\n");
        const MapObject * mo;
        bool shouldRun = false;
        for (std::vector<MapObject>::iterator it = data.areas.begin(); it != data.areas.end(); ++it)
        {
            mo = &(*it);
            if (mo->objType == MapObjectType_Load)
            {
                shouldRun = mo->nameHash == 0;
                if (shouldRun == false)
                {
                    shouldRun = Save::FlagIsSet(mo->nameHash);
                    FlagSet(mo->id, shouldRun);
                }
                if (shouldRun && mo->cmdCount > 0)
                {
                    Log_Info("Running sequence %i...\n", mo->id);
                    Sequencer::Run(new MapSequence(this, mo));
                }
            }
        }
    }

    Log_Info("Map loaded.\n\n");
    return true;
}

void TileMap::UpdateFlags()
{
    Log_Info("Updating map flags...\n");
    for (std::vector<MapObject>::iterator it = data.areas.begin(); it != data.areas.end(); ++it)
    {
        if (it->objType == MapObjectType_Load)
        {
            FlagSet(it->id, Save::FlagIsSet(it->nameHash));
        }
    }
    if (allDead == false)
    {
        allDead = true;
        for (std::vector<MapObject>::iterator it = data.entities.begin(); it != data.entities.end(); ++it)
        {
            if (MapEntityTypes::IsActor(&(*it)) == false)
                continue;

            if (it->entity != NULL && it->entity->IsEnabled() && it->entity->GetFaction() == ActorFaction_EVIL)
            {
                if (((Actor*)it->entity)->inventory.hearts > 0)
                {
                    allDead = false;
                    break;
                }
            }
        }
        if (allDead)
        {
            FlagSet(1, true);
            for (std::vector<MapObject>::iterator it = data.areas.begin(); it != data.areas.end(); ++it)
            {
                if (it->objType == MapObjectType_Trigger && it->flag && FlagIsSet(it->flag) && it->cmdCount > 0)
                {
                    Sequencer::Run(new MapSequence(this, &(*it)));
                    break;
                }
            }
        }
    }
}

void TileMap::ClearLayerMaterials()
{
    for (u8 i = 0; i < MAP_MESH_COUNT; i++)
    {
        layerMaterials[i] = NULL;
        if (layerMaterials[i] != NULL)
        {
            Material::Free(layerMaterials[i]);
        }
        meshes[i].Clear();
    }
    UpdateLayerMaterials();
}

void TileMap::SetLayerMaterial(u8 layer, LayerType layerType, Ref<Material> & material)
{
    u8 layerIndex = layer;
    if (layerType == LayerType_Wall)
    {
        layerIndex += MAP_LAYER_COUNT;
    }
    layerMaterials[layerIndex] = material;
    UpdateLayerMaterials();
}

void TileMap::SetLayerMaterial(u8 layer, LayerType layerType, const char * texturePath, const char * shaderPath)
{
    if (shaderPath == NULL || shaderPath[0] == '\0')
    {
        shaderPath = "shaders/map.dsh";
    }
    Ref<Material> & mat = Material::Load(texturePath, shaderPath);
    SetLayerMaterial(layer, layerType, mat);
}

void TileMap::UpdateLayerMaterials()
{
    for (u8 i = 0, lim = MAP_MESH_COUNT; i < lim; i++)
    {
        meshes[i].SetMaterial(layerMaterials[i]);
    }
}

bool TileMap::BuildTile(const TileInfo & tile, i16 x, i16 z)
{
    Direction rampDir = GetRampDir(tile.type);
    if (rampDir != Direction_None)
    {
        BuildRamp(tile, rampDir, x, z);
        return true;
    }
    
    if (tile.layer > 0 && tile.type != TileType_Empty)
    {
        BuildSurface(GetMesh(tile.layer, LayerType_Ground), tile.type, x, z, tile.layer);
    }
	
	if (IsCornerType(tile.type))
    {
		TileType underType = FlipTypeDir(tile.type);
		if (tile.under > 0)
        {
            BuildSurface(GetMesh(tile.under, LayerType_Ground), underType, x, z, tile.under);
        }
    }

    Mesh * wallMesh = GetMesh(tile.layer, LayerType_Wall);
    for (i16 i=1; i < 5; i++)
    {
        Direction dir = (Direction)i;
		i16 adjX = x + Directions::GetX(dir);
		i16 adjZ = z + Directions::GetZ(dir);
        const TileInfo & adjTile = GetTileInfo(adjX, adjZ);
		if (adjTile.under >= tile.layer)
			continue;

        if (IsCornerDir(tile.type, dir) == false && tile.layer != MAP_LAYER_WATER)
        {
            BuildWall(wallMesh, tile.type, x, z, dir, tile.layer, adjTile.under);
        }
    }
	
	if (IsCornerType(tile.type))
    {
		BuildWallCorner(wallMesh, tile.type, x, z, tile.layer, tile.under);
    }

    return true;
}

void TileMap::BuildSurface(Mesh * mesh, TileType type, i16 x, i16 z, u8 layer)
{
    float y = GetLayerHeight(layer);
    float off = data.tileSize * 0.5f + OVERFLOW_XZ;

    std::vector<Vector3> vertices;
    std::vector<Vector2> uvs;
    std::vector<Vector3> normals;
    std::vector<u32> triangles;

    if (type != TileType_CornerSE)
    {
        vertices.push_back(Vector3(x * data.tileSize - off, y, z * data.tileSize - off));
        uvs.push_back(Vector2(x, z));
        normals.push_back(Vector3_UP);
    }

    if (type != TileType_CornerSW)
    {
        vertices.push_back(Vector3(x * data.tileSize + off, y, z * data.tileSize - off));
        uvs.push_back(Vector2(x + 1, z));
        normals.push_back(Vector3_UP);
    }

    if (type != TileType_CornerNW)
    {
        vertices.push_back(Vector3(x * data.tileSize + off, y, z * data.tileSize + off));
        uvs.push_back(Vector2(x + 1, z + 1));
        normals.push_back(Vector3_UP);
    }

    if (type != TileType_CornerNE)
    {
        vertices.push_back(Vector3(x * data.tileSize - off, y, z * data.tileSize + off));
        uvs.push_back(Vector2(x, z + 1));
        normals.push_back(Vector3_UP);
    }

    mesh->AddTriangleFan(vertices, uvs, normals);
}

void TileMap::BuildWall(Mesh * mesh, TileType type, i16 x, i16 z, Direction direction, u8 layerMax, u8 layerMin)
{
    Vector3 forward = Directions::GetForward(direction);
	Vector3 right = glm::cross(forward, Vector3_UP);
	float off = data.tileSize * 0.5f;
	Vector3 origin = Vector3(x * data.tileSize, GetLayerHeight(layerMin), z * data.tileSize) + forward * off;
	Vector3 a = origin - right * off;
	Vector3 b = origin + right * off;
	Vector3 offsetTop = Vector3_UP * (GetLayerHeight(layerMax) - GetLayerHeight(layerMin));
	Vector3 c = b + offsetTop;
	Vector3 d = a + offsetTop;
	BuildQuad(mesh, a, b, c, d, forward);
}

void TileMap::BuildWallCorner(Mesh * mesh, TileType type, i16 x, i16 z, u8 layerMax, u8 layerMin)
{
    Vector3 forward = GetCornerForward(type);
	Vector3 right = glm::cross(forward, Vector3_UP);
	float off = HYP_0 * data.tileSize;
    Vector3 rightOff = right * off;
	
	Vector3 origin(x * data.tileSize, GetLayerHeight(layerMin), z * data.tileSize);
	Vector3 a = origin - rightOff;
	Vector3 b = origin + rightOff;
	Vector3 offsetLevel = Vector3_UP * (GetLayerHeight(layerMax) - GetLayerHeight(layerMin));
	Vector3 c = b + offsetLevel;
	Vector3 d = a + offsetLevel;
	BuildQuad(mesh, a, b, c, d, forward);
}

void TileMap::BuildRamp(const TileInfo & info, Direction direction, i16 x, i16 z)
{
	Vector3 forward = Directions::GetForward(direction);
	Vector3 right = glm::cross(forward, Vector3_UP);
	float offV = data.tileSize * 0.5f;
	
	Mesh * mesh = GetMesh(info.layer, LayerType_Ground);
	
	Vector3 origin = Vector3(x * data.tileSize, GetLayerHeight(info.under), z * data.tileSize) - forward * offV;
	Vector3 a = origin - right * offV;
	Vector3 b = origin + right * offV;
	Vector3 offF = forward * data.tileSize + Vector3_UP * (GetLayerHeight(info.layer) - GetLayerHeight(info.under));
	Vector3 c = b + offF;
	Vector3 d = a + offF;
	Vector3 normal = glm::normalize(Vector3_UP * data.tileSize + forward * data.tileHeight);
	BuildQuad(mesh, a, b, c, d, Vector3_UP);

    Mesh * wallMesh = GetMesh(info.layer, LayerType_Wall);
    BuildQuad(wallMesh, Vector3(d.x, GROUND_BOTTOM, d.z), Vector3(a.x, GROUND_BOTTOM, a.z), a, d, -right);
    BuildQuad(wallMesh, Vector3(c.x, GROUND_BOTTOM, c.z), Vector3(b.x, GROUND_BOTTOM, b.z), b, c, right);
}

void TileMap::BuildQuad(Mesh * mesh, Vector3 a, Vector3 b, Vector3 c, Vector3 d, Vector3 normal, Vector2 maxUv)
{
	std::vector<Vector3> vertices {a, b, c, d};
	std::vector<Vector2> uvs {Vector2(0, maxUv.y), Vector2(maxUv.x, maxUv.y), Vector2(maxUv.x, 0), Vector2_ZERO};
    std::vector<Vector3> normals {normal, normal, normal, normal};
    mesh->AddTriangleFan(vertices, uvs, normals);
}

void TileMap::BuildEdges()
{
    i16 padTiles = data.border * 2;

    static int OPS[] {
        -padTiles, data.width + padTiles, -padTiles, 0,
        -padTiles, data.width + padTiles, data.depth, data.depth + padTiles,
        -padTiles, 0, -padTiles, data.depth + padTiles,
        data.width, data.width + padTiles, -padTiles, data.depth + padTiles,
    };

    const TileInfo * info;
    for (size_t i = 0; i < 16; i += 4)
    {
        i16 x = OPS[i], xmax = OPS[i + 1], z = 0, zmin = OPS[i + 2], zmax = OPS[i + 3], tx, tz;
        for (; x < xmax; x++)
        {
            tx = x;
            for (z = zmin; z < zmax; z++)
            {
                tz = z;
                ClampOrds(&tx, &tz);
                info = &GetTileInfo(tx, tz);
                BuildTile(*info, x, z);
            }
        }
    }
}

Mesh * TileMap::GetMesh(u8 layer, LayerType layerType)
{
    u8 index = layer - 1;
    if (layerType == LayerType_Wall)
    {
        index += MAP_LAYER_COUNT;
    }
    return &meshes[index];
}
