#include "SwordGameScene.h"
#include <Material.h>
#include <Camera.h>
#include <Input.h>
#include <desslog.h>
#include <string.h>
#include <RenderWindow.h>
#include <Save.h>
#include <Shader.h>
#include <random>
#include <time.h>
#include <Maths.h>
#include <Party.h>
#include <AttackManager.h>
#include <Sequencer.h>
#include <Entity.h>
#include <MusicPlayer.h>
#include <fx/FXer.h>
#include <SwordGame.h>
#include <ui/UIOptionsMenuCtrl.h>
#include <MapEnvironment.h>

#ifdef DEBUG
#include <DebugOverlay.h>
#endif

#define OPTION_SECRET_PRESS_COUNT 20

enum PauseOption
{
    PauseOption_RESUME,
    PauseOption_OPTIONS,
    PauseOption_MUSIC,
    PauseOption_CREDITS,
    PauseOption_QUIT
};

void SwordGameScene::Init()
{   
    cam.backgroundColor = Vector3_ZERO;
    cam.SetCurrent();
    lookOffset = Vector3_ZERO;

    map.SetEntityFactory(&entityFactory);

    AddEntity(&map);
    AddEntity(&player);
    AddEntity(&ui);

    srand(time(NULL));

    player.map = &map;
    player.transform.SetPosition(Vector3(10, 1, 10));

    SettingsData & settings = Settings::Load();
    spawnFromSavePoint = settings.spawnFromSavePoint;

    std::string mapPath;
    u32 currentMap = spawnFromSavePoint ? Save::GetSaveMap() : Save::GetData()->map;
    if (currentMap != 0)
    {
        mapPath = MapUtils::MapHashToPath(currentMap);
    }
    if (mapPath.length() < 1)
    {
        mapPath = "maps/home.tmx";
    }
    Save::GetInventory(&player.inventory);
    gameTime = (float)Save::GetSeconds();
    if (player.inventory.maxHearts < PLAYER_DEFAULT_HEARTS)
    {
        player.inventory.maxHearts = PLAYER_DEFAULT_HEARTS;
    }
    if (player.inventory.hearts < player.inventory.maxHearts)
    {
        player.inventory.hearts = player.inventory.maxHearts;
    }
    map.Load(mapPath.c_str(), MapLoadFlag_Default, this);
    SetDefaultOptions();
    uiOptions.Added();
    uiProgress.Added();
    uiProgress.transform.position = Vector3(60, 50, 0);
    uiOptions.handler = NULL;
    showingOptions = false;
}

void SwordGameScene::SetDefaultOptions()
{
    uiOptions.ClearOptions();
    uiOptions.SetTitle("PAUSE");
    uiOptions.AddOption(TitleOption { PauseOption_RESUME, "Resume" });
    uiOptions.AddOption(TitleOption { PauseOption_OPTIONS, "Options" });
    if (optionPressCount >= OPTION_SECRET_PRESS_COUNT)
    {
        uiOptions.AddOption(TitleOption { PauseOption_MUSIC, "Music" });
        uiOptions.AddOption(TitleOption { PauseOption_CREDITS, "Credits" });
    }
    uiOptions.AddOption(TitleOption { PauseOption_QUIT, "Quit To Title" });
    uiOptions.handler = this;
    optionPressCount = 0;
}

void SwordGameScene::BeforeMapLoad()
{
    if (map.data.name == NULL || map.data.name[0] == '\0')
        return;
    
    Log_Info("Unloading map %s\n", map.data.name);
    Save::SetMap(map.data.name, &map.flags[0]);
    Save::SetSeconds((u32)gameTime);

    Log_Info("Saving position\n");
    const MapCmd * enterCmd = map.GetObjectCmd(map.GetArea(player.transform.position), MapCmd_ENTER);
    if (enterCmd != NULL)
    {
        const char * name = map.data.names[enterCmd->value].Text();
        Save::SetMapPrevious(name);
    }
    else
    {
        Save::SetMapPrevious(map.data.name);
    }

    Log_Info("Saving inventory\n");
    Save::SetInventory(&player.inventory);
    Save::SetPosition(player.exitOffset[0], player.exitOffset[1]);
    
    Log_Info("Unloading party\n");
    Party::Inst().Unload();
    Log_Info("Unloading attacks\n");
    AttackManager::Clear();
    Log_Info("Unloading FX\n");
    FXer::Clear();
    Log_Info("Unloading sequence\n");
    Sequencer::Stop();
#ifdef DEBUG
    Log_Info("Unloading environment\n");
    MapEnvironment::MarkUnloaded();
#endif
}

void SwordGameScene::OnMapLoaded(const char * mapPath)
{
    Log_Info("Loaded map %s\n", mapPath);

    Save::SetMapCurrent(map.data.name);
    Save::SaveFile();
    Save::GetMap(map.data.name, &map.flags[0]);

    Vector3 spawnPos = Vector3_ZERO;
    Direction walkDir = Direction_South;
    if (spawnFromSavePoint)
    {
        walkDir = Direction_None;
        const MapObject * saveObj = NULL;
        for (u16 i=0; i<map.data.entities.size(); i++)
        {
            saveObj = &map.data.entities[i];
            if (saveObj != NULL && Entity::NotDestroyed(saveObj->entity) && MapEntityTypes::ToType(saveObj->type) == MapEntityType_SaveSwitch)
            {
                spawnPos = map.GetPosition(saveObj->x, saveObj->z);
                break;
            }
            saveObj = NULL;
        }
        if (saveObj == NULL)
        {
            spawnPos = map.GetPosition(map.data.width / 2, map.data.depth / 2);
        }
    }
    else
    {
        u8 saveX, saveZ;
        Save::GetPosition(&saveX, &saveZ);

        const MapObject * prevArea = map.GetAreaByHash(Save::GetData()->prevMap);
        if (prevArea != NULL)
        {
            const MapCmd * cmdDir = map.GetObjectCmd(prevArea, MapCmd_DIR);
            if (cmdDir != NULL)
            {
                walkDir = (Direction)cmdDir->value;
            }
            i16 px = prevArea->x + prevArea->width / 2;
            i16 pz = prevArea->z + prevArea->depth / 2;
            i16 x, z;
            if (px < 1)
            {
                walkDir = Direction_East;
            }
            else if (px > map.data.width - 2)
            {
                walkDir = Direction_West;
            }
            else if (pz < 1)
            {
                walkDir = Direction_South;
            }
            else if (pz > map.data.depth - 2)
            {
                walkDir = Direction_North;
            }
            Vector3 enterDir = walkDir != Direction_None ? Directions::GetForward(walkDir) : player.GetDir();
            map.GetAreaEdge(prevArea, enterDir, x, z);
            if (walkDir == Direction_North || walkDir == Direction_South)
            {
                x = saveX + prevArea->x;
            }
            else if (walkDir == Direction_East || walkDir == Direction_West)
            {
                z = saveZ + prevArea->z;
            }
            prevArea->Constrain(x, z);
            spawnPos = map.GetPosition(x, z);
        }
        else
        {
            const MapObject * prevHut = map.GetEntityByName(mapLoaded);
            if (prevHut != NULL)
            {
                spawnPos = map.GetPosition(prevHut->x + 1, prevHut->z + 1, 0.0f);
            }
            else
            {
                spawnPos = map.GetPosition(map.data.width / 2, map.data.depth / 2);
            }
        }
    }
    player.Spawn(spawnPos, walkDir);
    strncpy(mapLoaded, MapUtils::MapPathToName(mapPath).c_str(), MAP_NAME_LENGTH);
    Log_Info("Loaded map as %s\n", mapLoaded);
    lookOffset = Vector3_ZERO;

    Party::Inst().SetHost(&player);
    Party::Inst().Load(&map);

    bool hasPlayedMusic = false;
    if (MapStr::ReadText(&map.data.music) != NULL)
    {
        hasPlayedMusic = MusicPlayer::Inst().Play(map.data.music.Text());
    }
    if (hasPlayedMusic == false)
    {
        MusicPlayer::Inst().Resume();
    }

    RenderWindow::Current()->SkipNext();
    spawnFromSavePoint = false;

    if (Save::FlagIsSet(SAVE_FLAG_ENDING) && Save::FlagIsSet(SAVE_FLAG_NEWGAMEPLUS) == false)
    {
        Save::FlagSet(SAVE_FLAG_NEWGAMEPLUS, true);
        GameUI::Current()->SetMessage("Somehow, after Blacklordoron was defeated, the enemies all grew in strength.", 3.0f, MessageType_Standard);
    }
}

void SwordGameScene::Update(float dt)
{
    if (map.IsLoading())
    {
        RenderWindow::Current()->SkipNext();
        map.LoadContinue();
        return;
    }

    Vector3 pos = Vector3_ZERO;
    float padX = 18.0f;
    float padZ = 16.0f;
    float offZ = 20.0f;
    for (u32 i = 0, lim = entities.size(); i < lim; i++)
    {
        if (entities[i]->entityOwner != (Entity*)&map)
            continue;

        if (entities[i]->IsAdded() && entities[i]->UseCulling())
        {
            pos = entities[i]->transform.position;
            entities[i]->SetVisible(
                pos.x > cam.transform.position.x - padX &&
                pos.x < cam.transform.position.x + padX &&
                pos.z > cam.transform.position.z - padZ - offZ &&
                pos.z < cam.transform.position.z + padZ - offZ
            );
        }
    }

    if (showingOptions)
    {
        uiOptions.Update(dt);
        if (showingOptions && Input::WasPressed(InputButton_BACK))
        {
            HideOptions();
        }
        if (optionPressCount < OPTION_SECRET_PRESS_COUNT && Input::WasPressed(InputButton_B))
        {
            optionPressCount += 1;
            if (optionPressCount >= OPTION_SECRET_PRESS_COUNT)
            {
                SetDefaultOptions();
            }
        }
    }
    else
    {
        Scene::Update(dt);
        gameTime += dt;
        Sequencer::Tick();

        if (showingOptions == false && (Input::WasPressed(InputButton_START) || Input::WasKeyPressed(SDL_SCANCODE_ESCAPE)))
        {
            showingOptions = true;
            Save::SetSeconds((u32)gameTime);
            SetDefaultOptions();
            uiProgress.OnEnabled();
            RenderWindow::Current()->SetPostShader("shaders/post-title.dsh");
        }
    }

    if (ui.isVisible == showingOptions)
    {
        ui.SetVisible(showingOptions == false);
    }

    Camera * cam = Camera::Current();
    cam->fov = 30.0f;

#ifdef DEBUG
    if (Input::IsKeyHeld('i')) lookOffset.z -= dt * 10.0f;
    if (Input::IsKeyHeld('k')) lookOffset.z += dt * 10.0f;
    if (Input::IsKeyHeld('j')) lookOffset.x -= dt * 10.0f;
    if (Input::IsKeyHeld('l')) lookOffset.x += dt * 10.0f;
    if (Input::WasKeyPressed(']'))
    {
        if (debugOverlay == NULL)
        {
            debugOverlay = new DebugOverlay();
        }
        else if (debugOverlay != NULL)
        {
            delete debugOverlay;
            debugOverlay = NULL;
        }
    }
#endif

    Vector3 targetPos = Vector3(player.transform.position);
    targetPos.y = 0.0f;
    Vector3 camPos = Vector3(targetPos);

    float dist = 20.0f / map.data.zoom;
    float heightMulti = 1.0f;
    bool shouldLock = cameraLocking;

    Vector3 camOffsetTo = camOffset;

    GameUI * ui = GameUI::Current();
    if (ui != NULL && ui->HasMessage())
    {
        if (messageWaitType == MessageWaitType_None)
        {
            if (ui->IsTalking())
            {
                messageWaitType = MessageWaitType_Talk;
                talkTarget = Sequencer::GetTarget();
            }
            else
            {
                messageWaitType = MessageWaitType_Message;
            }
        }
    }
    else
    {
        messageWaitType = MessageWaitType_None;
    }

    camOffsetTo.x = 0.0f;
    if (messageWaitType == MessageWaitType_Talk)
    {
        heightMulti = 0.75f;
        dist = 13.0f;
        if (talkTarget != NULL)
        {
            targetPos = talkTarget->transform.position;
        }
    }
    else if (messageWaitType == MessageWaitType_Message)
    {
        dist = 15.0f;
    }

    if (firstPerson)
    {
        Vector3 forward = player.transform.Forwards();
        targetPos.y += 1.3f;
        camOffsetTo = forward * 6.0f;
        camOffsetTo.y += 2.0f;
    }
    else
    {
        if (shouldLock)
        {
            float offz = 1.5f;
            targetPos.z -= offz;
            targetPos = map.ClampPosition(targetPos, camOffset.z * 0.45f);
            targetPos.z += offz;
            targetPos.y = camPos.y;
        }
        camOffsetTo.z = dist;
        camOffsetTo.y = dist * heightMulti;
    }

    camOffset = Maths::Lerp(camOffset, camOffsetTo, 0.3f, dt);
    camPos = targetPos + camOffset;

    cam->transform
        .SetPosition(camPos + lookOffset)
        .LookAt(targetPos + lookOffset);

#ifdef DEBUG
    if (Input::WasKeyPressed('v'))
    {
        firstPerson = firstPerson == false;
    }
    if (Input::WasKeyPressed('z'))
    {
        RenderWindow::Current()->SetRenderMode(
            RenderWindow::Current()->GetRenderMode() == RenderMode_Wireframe ?
                RenderMode_Standard :
                RenderMode_Wireframe
        );
        //Shader::ReloadAll();
    }
    if (Input::WasKeyPressed('l'))
    {
        cameraLocking = cameraLocking == false;
        if (cameraLocking)
        {
            lookOffset.x = lookOffset.y = lookOffset.z = 0.0f;
        }
    }
#endif

    cam->Update(dt);

    collisionChecker.Check(this);

    u8 newAttackCount = AttackManager::GetAttackerCount();
    if (newAttackCount > attackerCount || (attackIntensity > 0 && newAttackCount == 0))
    {
        attackerCount = newAttackCount;
        u16 totalHearts = 0;
        u8 entityHearts;
        for (EntityIt it = EntitiesBegin(); it != EntitiesEnd(); ++it)
        {
            if ((*it) != NULL && MapEntityTypes::IsBaddie(((Actor*)(*it))->type))
            {
                if ((*it)->IsEnabled() == false && (*it)->GetFaction() == ActorFaction_EVIL)
                    continue;

                if (Maths::IsWithinDistance(player.transform.position - (*it)->transform.position, 20.0f) == false)
                    continue;

                totalHearts += ((Actor*)(*it))->inventory.hearts;
            }
        }
        u8 newIntensity = (totalHearts + 2) / 3;
        if (newIntensity > 2)
        {
            newIntensity = 2;
        }
        if (newIntensity != attackIntensity)
        {
            if (newIntensity > attackIntensity)
            {
                attackIntensity = newIntensity;
                char fightMusic[8];
                sprintf(fightMusic, "fight%i", attackIntensity + 1);
                MusicPlayer::Inst().PlayOverride(MusicPlayer::Inst().GetCue(fightMusic));
            }
            else if (newIntensity == 0)
            {
                attackIntensity = 0;
                MusicPlayer::Inst().StopOverride();
            }
        }
    }

    MusicPlayer::Inst().Update(dt);

    if (player.wantMap[0] != '\0')
    {
        char mapPath[32];
        sprintf(mapPath, "maps/%s.tmx", player.wantMap);
        player.wantMap[0] = '\0';
        player.Reset();
        Log_Info("Player requested map %s\n", mapPath);
        map.Load(mapPath, MapLoadFlag_Default, this);
        RenderWindow::Current()->SkipNext();
    }
}

void SwordGameScene::OptionSelect(u8 index)
{
    switch (index)
    {
        case PauseOption_RESUME:
            HideOptions();
            break;
            
        case PauseOption_OPTIONS:
            optionsCtrl = new UIOptionsMenuCtrl(&uiOptions);
            break;
            
        case PauseOption_MUSIC:
        {
            u8 count = MusicPlayer::Inst().GetCueCount();
            u8 index = MusicPlayer::Inst().GetCueIndex();
            index = (index + 1) % count;
            HashInt nextHash = MusicPlayer::Inst().GetCueAt(index);
            if (nextHash != 0)
            {
                MusicPlayer::Inst().Play(MusicPlayer::Inst().GetCue(nextHash));
            }
            break;
        }
        
        case PauseOption_CREDITS:
            SwordGame::Inst().Credits();
            break;
        
        case PauseOption_QUIT:
            ((SwordGame*)game)->ToTitle();
            break;
    }
}

void SwordGameScene::OptionBack()
{
    SetDefaultOptions();
}

void SwordGameScene::HideOptions()
{
    showingOptions = false;
    RenderWindow::Current()->SetPostShader(NULL);
    uiOptions.handler = NULL;
}

void SwordGameScene::Render()
{
    Scene::Render();
    if (showingOptions)
    {
        uiOptions.Render();
        uiProgress.Render();
    }
#ifdef DEBUG
    if (debugOverlay != NULL)
    {
        debugOverlay->Render(this);
    }
#endif
}

void SwordGameScene::OnUnload()
{
    Sequencer::Stop();
    AttackManager::Clear();
    Party::Inst().Unload();
}
