#include <Player.h>
#include <Input.h>
#include <Camera.h>
#include <Maths.h>
#include <props/PushableProp.h>
#include <ui/GameUI.h>
#include <desslog.h>
#include <props/PropDoor.h>
#include <props/PropCupboard.h>
#include <SwordGame.h>
#include <fx/FXStrikeSword.h>
#include <AttackManager.h>
#include <Sequencer.h>
#include <MapSequence.h>
#include <props/PropSwitch.h>
#include <fx/FXer.h>
#include <Save.h>
#include <MusicPlayer.h>
#include <SwordGameScene.h>

const InputButton BUTTON_STRIKE = InputButton_A;
const InputButton BUTTON_ROLL = InputButton_B;

Player::Player() : Actor(MapEntityType_Player)
{
    faction = ActorFaction_PLAYER;
    wantMap[0] = '\0';
}

Player::~Player()
{
}

void Player::Start()
{
    SetHoldItem(inventory.holdItem);
    Actor::Start();
    input = Vector2_ZERO;
    moveSpeed = 8.0f;
    maxGradient = 0.72f; 
}

void Player::LoadModel()
{
    const char * intendedModel = inventory.holdItem == MapEntityType_SwordParticulars ? 
        "models/blacksmithdress.iqm" :
        "models/blacksmith.iqm";
    if (Hash(intendedModel) != model.modelHash)
    {
        model.Clear();
        model.LoadModel(intendedModel, "models/blacksmith.iqm", 0.34f);
        model.SetMaterial("models/blacksmith.png", ACTOR_SHADER_PATH);
    }
    Actor::LoadModel();
}

void Player::Spawn(Vector3 position)
{
    MapEntity::Spawn(position);
    insideArea = NULL;
    map->GetTile(position, &areaPoint.x, &areaPoint.z);
    GameUI * ui = GameUI::Current();
    if (ui != NULL)
    {
        ui->HideMessages();
    }
    walkPoint = WalkPoint_NONE;
    SetHoldItem(inventory.holdItem);
}

void Player::Spawn(Vector3 position, Direction walkDir)
{
    wantMap[0] = '\0';
    spawnWalkDir = walkDir;
    Spawn(position);
    WalkIn();
}

void Player::Respawn()
{   
    wantMap[0] = '\0';
    MapEntity::Respawn();
    map->GetTile(transform.position, &areaPoint.x, &areaPoint.z);
    enterArea = map->GetArea(transform.position);
    insideArea = enterArea;
    WalkIn();
}

void Player::WalkIn()
{
    if (spawnWalkDir != Direction_None)
    {
        map->GetTile(transform.position, &walkPoint.x, &walkPoint.z);
        Directions::Move(spawnWalkDir, &walkPoint.x, &walkPoint.z, 1);
    }
}

Vector3 Player::GetMoveVelocity(float dt)
{
    if (inventory.hearts == 0 || inCupboard)
    {
        input = Vector2_ZERO;
        return Vector3_ZERO;
    }

    if (walkPoint != WalkPoint_NONE)
    {
        Vector3 autoWalkDiff = map->GetPosition(walkPoint.x, walkPoint.z) - transform.position;
        autoWalkDiff.y = 0.0f;
        if (Maths::LengthSq(autoWalkDiff) < 0.5f)
        {
            walkPoint = WalkPoint_NONE;
        }
        else
        {
            input = Vector2_ZERO;
            return Maths::Clamp(autoWalkDiff * 50.0f, moveSpeed);
        }
    }

    GameUI * ui = GameUI::Current();
    if (ui != NULL && ui->HasMessage())
    {
        input = Vector2_ZERO;
        return Vector3_ZERO;
    }

    input = Input::GetStick(InputStick_LEFT);

#ifdef DEBUG
    if (fall >= 0.0f && Input::WasKeyPressed('f'))
    {
        fall = -10.0f;
    }
#endif

    Vector3 moveVelocityTo = Vector3_ZERO;

    Camera * cam = Camera::Current();
    bool hasInput = Maths::LengthSq(input) > 0.1f;
    if (hasInput)
    {
        Vector3 right = cam->transform.Right();
        Vector3 forwards = Maths::Cross(right, Vector3(0, 1, 0));
        moveVelocityTo += right * input.x;
        moveVelocityTo += forwards * input.y;
        moveVelocityTo = Maths::Clamp(moveVelocityTo, 1.0f) * 8.0f;
    }
    
    if (animDownTime > 0.0f)
    {
        moveVelocityTo *= animMoveMulti;
    }

    return moveVelocityTo;
}

void Player::Update(float dt)
{
    MapEntity::Update(dt);
    GameUI * ui = GameUI::Current();
    if (ui != NULL)
    {
        ui->SetInventory(&inventory);
    }

    promptTime += dt;

    wantsStrike = Input::WasPressed(BUTTON_STRIKE);
    wantsRoll = Input::WasPressed(BUTTON_ROLL);
    
    if (ui->IsWriting())
    {
        AttackManager::Chill();
        timeInvincible = 0.5f;
        if (promptTime > 0.1f && wantsStrike)
        {
            ui->FinishWriting();
            promptTime = 0.0f;
            wantsStrike = false;
        }
        slipVelocity = Vector3_ZERO;
        return;
    }

    if (Sequencer::IsRunning())
    {
        AttackManager::Chill();
        input = Vector2_ZERO;
        if (Sequencer::IsPaused())
        {
            timeInvincible = 0.5f;
            if (ui->GetMessageStatus() == MessageStatus_Pending)
            {
                ui->SetConfirmPrompt("Yes", "No");
                if (wantsStrike)
                {
                    ui->SetMessageStatus(MessageStatus_Confirm);
                }
                else if (wantsRoll)
                {
                    ui->SetMessageStatus(MessageStatus_Cancel);
                }
            }
            else
            {
                ui->SetPrompt("Continue");
                if (wantsStrike || wantsRoll)
                {
                    ui->SetMessageStatus(MessageStatus_Confirm);
                }
            }
            if (ui->GetMessageStatus() >= MessageStatus_Confirm)
            {
                ui->HideMessages();
                Sequencer::Resume();
                wantsStrike = false;
                wantsRoll = false;
            }
        }
        slipVelocity = Vector3_ZERO;
        return;
    }


    if (ui->HasMessage())
    {
        AttackManager::Chill();
        timeInvincible = 0.5f;
        ui->SetPrompt("OK");
        if (wantsStrike)
        {
            ui->HideMessages();
            wantsStrike = false;
        }
        slipVelocity = Vector3_ZERO;
        return;
    }

    if (inCupboard)
    {
        AttackManager::Chill();
        dirMove = inCupboard->transform.Forwards();
        if (inCupboard->IsOpen())
        {
            animDownTime -= dt;
            if (animDownTime < 0.0f)
            {
                UsePrompt(inCupboard->info);
            }
        }
        if (wantsStrike)
        {
            SetVisible(true);
            inCupboard->SetOpen(true);
            transform.position = inCupboard->transform.position - inCupboard->transform.Forwards() * 1.5f;
            inCupboard = NULL;
            moveVelocity = dirMove;
            wantsStrike = false;
            collider->mode = CollisionMode_Resolve;
        }
        else
        {
        input = Vector2_ZERO;
        moveVelocity = Vector3_ZERO;
        return;
        }
    }

    if (promptObject != NULL)
    {
        if (Entity::NotDestroyed(promptObject->entity))
        {
            Face(promptObject->entity);
        }
        if (wantsStrike)
        {
            wantsStrike = false;
            UsePrompt(promptObject);
            return;
        }
        if (promptObject != insideArea && areaPoint != point)
        {
            promptObject = NULL;
            ui->HideMessages();
            areaPoint = MapPoint_MIN;
        }
    }

    if (map != NULL)
    {
        const MapObject * area = map->GetArea(transform.position);
        if (area != NULL && area != insideArea)
        {
            if (insideArea == NULL)
            {
                Log_Info("enter area %s\n", area->name.Text());
            }
            insideArea = (MapObject *)area;
            promptObject = NULL;
            if (ui != NULL)
            {
                ui->HideMessages();
            }
            if (point != areaPoint)
            {
                if (area->objType == MapObjectType_Trigger)
                {
                    UsePrompt(area);
                }
                if (area->objType == MapObjectType_SpawnPoint)
                {
                    Save::SetSpawnPoint(area->nameHash);
                    spawnPos = map->GetPosition(area->x + area->width / 2, area->z + area->depth / 2);
                }
                else if (area->objType == MapObjectType_MessageArea)
                {
                    SetPrompt("Inspect", area);
                }
                areaPoint = point;
                areaEnterPos = transform.position;
            }
        }
        else if (area == NULL)
        {
            if (insideArea != NULL)
            {
                if (enterArea != NULL && enterArea != insideArea && insideArea->objType == MapObjectType_MapArea)
                {
                    GotoMap(insideArea);
                }
                insideArea = NULL;
                enterArea = NULL;
                promptObject = NULL;
                areaPoint = point;
                GameUI * ui = GameUI::Current();
                if (ui != NULL)
                {
                    ui->HideMessages();
                }
            }
        }

        if (insideArea != NULL && insideArea != enterArea && insideArea->objType == MapObjectType_MapArea)
        {
            if (Maths::IsWithinDistance(transform.position - areaEnterPos, 1.5f) == false)
            {
                GotoMap(insideArea);
            }
        }
    }

    if (point != areaPoint)
    {
        areaPoint = MapPoint_MIN;
    }

    if (isGrounded && mapTile.type == TileType_Solid)
    {
        if (groundSafePointCandidate != point)
        {
            if (groundSafePointCandidate != MapPoint_MIN)
            {
                if (map->GetHeight(point.x, point.z) == map->GetHeight(groundSafePointCandidate.x, groundSafePointCandidate.z))
                {
                    groundSafePoint = groundSafePointCandidate;
                }
                else
                {
                    groundSafePoint = groundSafePointCandidate = MapPoint_MIN;
                }
            }
            groundSafePointCandidate = point;
        }
    }

    if (Maths::LengthSq(transform.position - prevPos) > 1.0f)
    {
        const char * sound = oddStep ? "sfx/step1.wav" : "sfx/step2.wav";
        //FXer::Fire(this, FXType_Dust);
        oddStep = !oddStep;
        SoundPlayer::Inst().Play(sound, 0.8f);
        prevPos = transform.position;
    }
}

void Player::OutOfMap()
{
    if (inventory.hearts > 1)
    {
        inventory.hearts--;
        RenderWindow::Current()->Blank(100);
        //Respawn();
        if (groundSafePoint != MapPoint_MIN)
        {
            transform.position = map->GetPosition(groundSafePoint.x, groundSafePoint.z);
        }
        else
        {
            Respawn();
        }
    }
    else
    {
        SetEnabled(false);
        OnDead();
    }
}

void Player::OnCollision(Entity * candidate)
{
    if (inCupboard)
        return;

    Actor::OnCollision(candidate);

    if (walkPoint != WalkPoint_NONE)
    {
        walkPoint = WalkPoint_NONE;
    }

    MapEntity * other = static_cast<MapEntity*>(candidate);
    if (other == NULL)
        return;

    if (other->GetFaction() == ActorFaction_EVIL)
    {
        if (other->GetDamage() > 0)
        {
            TakeHit(candidate, other->GetDamage());
            AttackManager::Chill();
            return;   
        }
    }

    MapEntityType otherType = other->GetType();
    if (otherType > MapEntityType_MAX)
        return;

    if (MapEntityTypes::IsPickup(otherType))
    {
        // pickups
        other->Remove();
        if (otherType == MapEntityType_DragonCoin)
        {
            SoundPlayer::Inst().Play("sfx/sword.wav", 0.5f);
            GameUI * ui = GameUI::Current();
            if (ui != NULL)
            {
                ui->SetMessage("You got a DragonCoin!\nPause to check it out!", 3.0f, MessageType_Silent);
                PlayAnimation(ActorAnim_Get, 1.0f, 10.0f);
                dirMove = Vector3_SOUTH;
                moveVelocity = Vector3_SOUTH;
                input = Vector2_ZERO;
                Save::FlagSet(other->info->nameHash, true);
                Save::SaveFile();
            }   
        }
        if (MapEntityTypes::IsSword(otherType))
        {
            SetHoldItemAnnounced(otherType);
            if (other->info != NULL && other->info->cmdCount > 0)
            {
                UsePrompt(other->info);
            }
        }
        else
        {
            map->FlagSet(other->info->id, true);
        }
    }

    if (MapEntityTypes::IsNPC(otherType))
    {
        SetPrompt("Talk", other->info);
        return;
    }
    if (MapEntityTypes::IsDoor(otherType))
    {
        if (inventory.holdItem != MapEntityType_Key)
        {
            SetPrompt("Open", other->info);
        }
        return;
    }
    if (otherType == MapEntityType_Sign)
    {
        SetPrompt("Read", other->info);
        return;
    }

    // push / interact
    Vector3 diff = other->transform.position - transform.position;
    float dot = Maths::Dot(dirMove, Maths::Normalize(diff));
    if (dot > 0.5f && Maths::Abs(diff.y) < 0.7f)
    {
        if (MapEntityTypes::IsPushable(otherType))
        {
            ((PushableProp*)other)->Push(Maths::Cardinal(dirMove));
            model.BlendAnimation(anims[ActorAnim_Push], 0.08f);
            animDownTime = 0.3f;
            animMoveMulti = 0.2f;
            slipVelocity = Vector3_ZERO;
            return;
        }
    }

    switch (otherType)
    {
        case MapEntityType_Hut:
        {
            Vector3 hutFront = other->transform.position;
            hutFront.z += 1.5f;
            if (Maths::Dot(dirMove, Vector3_NORTH) > 0.5f && Maths::LengthSq(hutFront - transform.position) < 3.0f)
            {
                GotoMap(other->info);
            }
            return;
        }
        case MapEntityType_Cupboard:
        {
            SetPrompt("Use", other->info);
            return;
        }
    }
}

bool Player::TakeHit(Entity * other, u8 power)
{
    if (Actor::TakeHit(other, power) == false)
        return false;
    AttackManager::Chill();
    Camera * cam = Camera::Current();
    if (cam != NULL)
    {
        cam->Shake(power);
    }
    if (Actor::IsActor(other))
    {
        ((Actor*)other)->Knock(this);
    }
    SoundPlayer::Inst().Play("sfx/hurt2.wav");
    return true;
}

void Player::Strike()
{
    if (inventory.holdItem == MapEntityType_SwordParticulars)
    {
        model.PlayAnimation(anims[ActorAnim_Pose], 0.0, 30.0f);
        animDownTime = 1.0f;
        animMoveMulti = 0.0f;
        timeInvincible = 1.0f;
        dirMove = Vector3_SOUTH;
        moveVelocity = Vector3_SOUTH;
        input = Vector2_ZERO;
        FXer::Fire(this, FXType_Love);
        return;
    }
    Actor::Strike();
}

void Player::OnStrike(MapEntity * other, u8 power)
{
    if (other == NULL)
        return;

    if (inventory.holdItem == MapEntityType_Axe)
    {
        if (other->type == MapEntityType_Tree)
        {
            other->TakeHit(this, 1);
        }
    }

    if (inventory.holdItem != MapEntityType_None && other->type == MapEntityType_Grass)
    {
        other->TakeHit(this, 1);
    }

    if (inventory.holdItem == MapEntityType_Key)
    {
        if (MapEntityTypes::IsDoor(other->type))
        {
            ((PropDoor*)other)->Open();
        }
    }

    if (inventory.holdItem == MapEntityType_SwordFire)
    {
        if (MapEntityTypes::IsFlammable(other->type))
        {
            other->Remove();
        }
    }

    if (inventory.holdItem == MapEntityType_SwordElectric)
    {
        if (other->type == MapEntityType_Obelisk)
        {
            if (((PropSwitch*)other)->IsOn() == false)
            {
                ((PropSwitch*)other)->SetOn(true);
                SetHoldItem(MapEntityType_Sword);
                GameUI * ui = GameUI::Current();
                if (ui != NULL)
                {
                    ui->SetMessage("Somehow, the power is drained from your sword.", 3.0f);
                    PlayAnimation(ActorAnim_Get, 1.0f, 15.0f);
                }
            }
        }
    }

    Actor::OnStrike(other, power);
}

void Player::GotoMap(const MapObject * mo)
{
    if (inventory.hearts == 0)
        return;
    strncpy(wantMap, MapStr::ReadText(&mo->name), MAP_NAME_LENGTH);
    promptObject = NULL;
    i16 x, z;
    map->GetTile(transform.position, &x, &z);
    exitOffset[0] = Maths::Clamp(x - mo->x, 0, mo->width);
    exitOffset[1] = Maths::Clamp(z - mo->z, 0, mo->depth);
}

void Player::SetPrompt(const char * text, const MapObject * obj)
{
    if (inventory.holdItem == MapEntityType_SwordParticulars || inCupboard)
    {
        promptObject = NULL;
        areaPoint = point;
        return;
    }
        
    if (obj != promptObject)
    {
        GameUI * ui = GameUI::Current();
        if (ui != NULL)
        {
            promptObject = obj;
            ui->SetPrompt(text);
            areaPoint = point;
        }
    }
}

void Player::UsePrompt(const MapObject * obj)
{
    if (obj == NULL)
        return;

    Log_Info("Player use %i\n", obj->id);
    
    if (obj->entity != NULL && inventory.holdItem != MapEntityType_Key)
    {
        MapEntityType type = MapEntityTypes::ToType(obj->type);
        if (type == MapEntityType_Cupboard)
        {
            inCupboard = (PropCupboard*)obj->entity;
            if (inCupboard->IsOpen())
            {
                SetVisible(false);
                transform.position = obj->entity->transform.position;
                inCupboard->SetOpen(false);
                promptObject = NULL;
            }
            else
            {
                inCupboard->SetOpen(true);
                animDownTime = 0.2f;
                collider->mode = CollisionMode_None;
                return;
            }
            dirMove = obj->entity->transform.Forwards();
        }
        else if (MapEntityTypes::IsDoor(type))
        {
            PropDoor * door = (PropDoor*)(obj->entity);
            if (door->IsLocked())
            {
                if (door->ShouldBeUnlocked() || door->ShouldBeOpen())
                {
                    door->Open();
                }
                else
                {
                    GameUI * ui = GameUI::Current();
                    if (ui != NULL)
                    {
                        ui->SetMessage("It's locked.", 2.0f);
                    }
                }
            }
            else if (door->IsOpen() == false)
            {
                door->Open();
            }
        }
    }
    if (obj->cmdCount > 0)
    {
        Sequencer::Run(new MapSequence(map, obj));
        promptTime = 0.0f;
    }
}

void Player::SetHoldItem(MapEntityType holdItem)
{
    if (holdItem == MapEntityType_Heart)
    {
        inventory.maxHearts += 2;
        inventory.hearts = inventory.maxHearts;
        return;
    }
    if (holdItem == MapEntityType_HeartPiece)
    {
        inventory.maxHearts += 1;
        inventory.hearts = inventory.maxHearts;
        return;
    }
    if (holdItem == MapEntityType_SwordParticulars)
    {
        if (holdItem != inventory.holdItem)
        {
            Actor::SetHoldItem(MapEntityType_None);
            inventory.holdItem = MapEntityType_SwordParticulars;
        }
    }
    else
    {
        Actor::SetHoldItem(holdItem);
    }
    if (model.IsLoaded())
    {
        LoadModel();
    }
}

void Player::SetHoldItemAnnounced(MapEntityType holdItem)
{
    if (holdItem == inventory.holdItem)
    {
        return;
    }
    SetHoldItem(holdItem);
    if (MapEntityTypes::IsSword(holdItem))
    {
        if (MusicPlayer::Inst().IsPersistant() == false)
        {
            MusicPlayer::Inst().PauseFor(2.5f);
            SoundPlayer::Inst().Play("sfx/sword.wav", 0.5f);
        }
        else
        {
            SoundPlayer::Inst().Play("sfx/sword.wav", 0.3f);
        }
        GameUI * ui = GameUI::Current();
        if (ui != NULL)
        {
            char msg[48];
            sprintf(msg, "You got the Sword of %s!\0", MapEntityTypes::GetSwordName(holdItem));
            ui->SetMessage(msg, 3.0f, MessageType_Silent);
            if (holdItem == MapEntityType_SwordParticulars)
            {
                PlayAnimation(ActorAnim_Pose, 1.0f, 10.0f);
            }
            else
            {
                PlayAnimation(ActorAnim_Get, 1.0f, 10.0f);
            }
            dirMove = Vector3_SOUTH;
            moveVelocity = Vector3_SOUTH;
            input = Vector2_ZERO;
        }   
    }
    else if (holdItem == MapEntityType_Heart)
    {
        GameUI * ui = GameUI::Current();
        if (ui != NULL)
        {
            ui->SetMessage("You got a whole new heart! No idea whose...\0", 3.0f);
            PlayAnimation(ActorAnim_Get, 1.0f, 15.0f);
            dirMove = Vector3_BACK;
            moveVelocity = Vector3_BACK;
            input = Vector2_ZERO;
        }
    }
    else if (holdItem == MapEntityType_HeartPiece)
    {
        GameUI * ui = GameUI::Current();
        if (ui != NULL)
        {
            ui->SetMessage("You got half a heart! Additional half-hearted efforts abound!\0", 3.0f);
            PlayAnimation(ActorAnim_Get, 1.0f, 15.0f);
            dirMove = Vector3_BACK;
            moveVelocity = Vector3_BACK;
            input = Vector2_ZERO;
        }
    }
}

void Player::OnDead()
{
    Save::LoadFile();
    Save::AddDeath();
    u32 time = (u32)((SwordGameScene*)scene)->GetTime();
    Save::SetSeconds(time);
    Save::SaveFile();
    SwordGame::Inst().Died();
}
