#include "Party.h"
#include <Save.h>
#include <MapEntity.h>
#include <actors/NPCActor.h>
#include <behaviours/FollowBehaviour.h>
#include <Hash.h>
#include <Scene.h>
#include <TileMap.h>

Party & Party::Inst()
{
    static Party instance;
    return instance;
}

HashInt Party::ToSaveFlag(MapEntityType memberType)
{
    return Hash("party") + (HashInt)memberType;
}

bool Party::IsMemberInParty(MapEntityType memberType)
{
    return Save::FlagIsSet(ToSaveFlag(memberType));
}

void Party::SetHost(Entity * value)
{
    host = value;
}

void Party::Load(TileMap * map)
{
    currentMap = map;
    for (u16 t=MapEntityType_NPCMin; t <= MapEntityType_NPCMax; t++)
    {
        if (IsMemberInParty((MapEntityType)t))
        {
            AddPartyMember((MapEntityType)t);
        }
    }
}

void Party::Unload()
{
    for (std::vector<NPCActor*>::iterator it = members.begin(); it != members.end(); ++it)
    {
        (*it)->Destroy();
    }
    members.clear();
}

bool Party::AddPartyMember(MapEntityType type)
{
    u8 index = GetMemberIndex(type);
    if (index < members.size())
        return false;

    NPCActor * npc = (NPCActor*)factory.Create(type);
    npc->type = type;
    return AddPartyMember(npc);
}

bool Party::AddPartyMember(NPCActor * member)
{
    u8 index = GetMemberIndex(member->type);
    if (index < members.size())
        return false;

    Save::FlagSet(ToSaveFlag(member->type), true);

    Entity * npcHost = index == 0 ? host : members[index - 1];
    member->SetBehaviour(new FollowBehaviour(member, npcHost, 6.0f));
    if (member->map != currentMap)
    {
        members.push_back(member);
        member->transform.position = npcHost->transform.position - npcHost->transform.Forwards();
        member->transform.rotation = npcHost->transform.rotation;
        currentMap->scene->AddEntity(member);
        member->map = currentMap;

        const MapObject * existingObj = currentMap->GetEntityInfo(member->type);
        if (existingObj != NULL && existingObj->entity != NULL)
        {
            existingObj->entity->Remove();
            ((MapObject*)existingObj)->entity = member;
        }
    }
    member->removeOnDeath = false;
    return true;
}

bool Party::RemovePartyMember(MapEntityType type)
{
    bool found = false;
    for (std::vector<NPCActor*>::iterator it = members.begin(); it != members.end(); ++it)
    {
        if ((*it)->type == type)
        {
            (*it)->SetBehaviour(NULL);
            (*it)->SetCollision(CollisionMode_Collide);
            found = true;
            break;
        }
    }
    
    Save::FlagSet(ToSaveFlag(type), false);
    
    if (!found)
    {
        return false;
    }

    return true;
}

u8 Party::GetMemberIndex(MapEntityType type) const
{
    u8 max = members.size();
    for (u8 i = 0; i < max; i++)
    {
        if (members[i]->type == type)
            return i;
    }
    return max;
}