#include "MusicPlayer.h"
#include <desslibs.h>
#include <Save.h>
#include <read/tinyxml2.h>
#include <loaders/TiledUtils.h>
#include <FileSystem.h>

using namespace tinyxml2;

MusicPlayer & MusicPlayer::Inst()
{
    static MusicPlayer inst;
    ModMusicPlayer::current = &inst;
    return inst;
}

MusicPlayer::MusicPlayer() {}
MusicPlayer::~MusicPlayer() {}

bool MusicPlayer::Play(const char * musicCmd)
{
    if (musicCmd != NULL && musicCmd[0] != '\0')
    {
        char firstChar = musicCmd[0];
        if (firstChar >= '0' && firstChar <= '9')
        {
            ModMusicCue cue;
            TiledUtils::ExtractMusicArgs(musicCmd, &cue.start, &cue.end, &cue.repeat);
            return ModMusicPlayer::Play(&cue);
        }
        else
        {
            const ModMusicCue * cue = MusicPlayer::Inst().GetCue(musicCmd);
            if (cue != NULL)
            {
                return ModMusicPlayer::Play(cue);
            }
        }
    }
    return ModMusicPlayer::Play(musicCmd);
}

void MusicPlayer::Resume()
{
    for (std::vector<ModMusicCue>::reverse_iterator it = cues.rbegin(); it != cues.rend(); ++it)
    {
        if (Save::FlagIsSet(it->hash))
        {
            ModMusicPlayer::Play(&(*it));
            return;
        }
    }
}

u8 MusicPlayer::GetCueCount() const
{
    return cues.size();
}

u8 MusicPlayer::GetCueIndex() const
{
    if (tracks[MusicTrack_Main].file != NULL)
    {
        for (u8 i = 0; i < cues.size(); i++)
        {
            if (cues[i].hash == tracks[MusicTrack_Main].cue.hash)
                return i;
        }
    }
    return 0;
}

HashInt MusicPlayer::GetCueAt(u8 index) const
{
    if (index >= cues.size())
        return 0;
    return cues[index].hash;
}

void MusicPlayer::Load(const char * musicXmlPath)
{
    files.clear();
    cues.clear();
    
    if (SDL_Init(SDL_INIT_AUDIO) < 0) 
    {
        Log_Error("Couldn't initialize SDL: %s\n", SDL_GetError());
        return;
    }

    XMLDocument doc;
    FileSystem::LoadXML(musicXmlPath, &doc);
    if (doc.Error())
    {
        Log_Error("Couldn't load music cues from %s\n", musicXmlPath);
        return;
    }
    
    XMLElement * rootNode = doc.FirstChildElement("music");
    if (rootNode == NULL)
    {
        Log_Error("Couldn't find valid cues in file %s\n", musicXmlPath);
        return;
    }

    XMLElement * cueNode = rootNode->FirstChildElement("cue");
    while (cueNode != NULL)
    {
        const char * modPath = cueNode->Attribute("file");
        ModMusicFile * file = LoadMod(modPath);
        ModMusicCue cue;
        cue.hash = Hash(cueNode->Attribute("save"));
        cue.fileHash = file->hash;
        cue.persistant = cueNode->IntAttribute("persistant", 0) > 0;
        const char * musicArgs = cueNode->Attribute("music");
        if (musicArgs != NULL)
        {
            TiledUtils::ExtractMusicArgs(musicArgs, &cue.start, &cue.end, &cue.repeat);
        }
        cues.push_back(cue);
        cueNode = cueNode->NextSiblingElement();
    }
}