#include <Missions.h>
#include <Save.h>
#include <read/tinyxml2.h>
#include <Hash.h>
#include <ui/GameUI.h>
#include <MusicPlayer.h>
#include <SoundPlayer.h>
#include <FileSystem.h>

using namespace tinyxml2;

Missions & Missions::Inst()
{
    static Missions inst;
    return inst;
}

void Missions::Load()
{
    XMLDocument doc;
    FileSystem::LoadXML("data/missions.xml", &doc);
    if (doc.Error())
    {
        Log_Error("Couldn't load missions");
        return;
    }
    
    XMLElement * missionsNode = doc.FirstChildElement("missions");
    if (missionsNode == NULL)
    {
        Log_Error("Couldn't find valid missions in file.");
        return;
    }

    XMLElement * missionNode = missionsNode->FirstChildElement("mission");
    while (missionNode != NULL)
    {
        const char * id = missionNode->Attribute("save");
        const char * name = missionNode->Attribute("name");
        const char * text = missionNode->Attribute("text");
        const char * sound = missionNode->Attribute("sound");
        Mission mission;
        mission.hash = Hash(id);
        strncpy(mission.name, name, MAP_NAME_LENGTH);
        strncpy(mission.text, text, MAP_MESSAGE_LENGTH);
        strncpy(mission.sound, sound, MAP_PATH_LENGTH);
        missions.push_back(mission);
        missionNode = missionNode->NextSiblingElement();
    }

    hasLoaded = true;
}

void Missions::Complete(const char * missionId)
{
    if (hasLoaded == false)
        Load();
    HashInt missionHash = Hash(missionId);
    Save::FlagSet(missionHash, true);
    Mission * mission = NULL;
    for (std::vector<Mission>::iterator it = missions.begin(); it != missions.end(); ++it)
    {
        if (it->hash == missionHash)
        {
            GameUI * ui = GameUI::Current();
            if (ui != NULL)
            {
                const char * name = it->name == NULL || it->name[0] == '\0' ? "QUEST!" : it->name;
                ui->SetMessage(name, it->text, 3.0f, MessageType_Silent);
            }
            MusicPlayer::Inst().PauseFor(5.0f);
            SoundPlayer::Inst().Play(it->sound, 0.3f, SoundFlag_No_Interupt);
        }
    }
}

MissionState Missions::GetState(const char * missionId)
{
    if (hasLoaded == false)
        Load();
    HashInt missionHash = Hash(missionId);
    for (std::vector<Mission>::iterator it = missions.begin(); it != missions.end(); ++it)
    {
        if (it->hash == missionHash)
        {
            if (Save::FlagIsSet(it->hash))
            {
                return MissionState_COMPLETE;
            }
            else
            {
                return MissionState_INCOMPLETE;
            }
        }
    }
    return MissionState_NONE;
}

GameProgress Missions::GetProgress() const
{
    GameProgress progress;
    progress.time = Save::GetSeconds();
    u8 coinCount = 0;
    for (u8 i=0; i<GameProgress_COIN_COUNT; i++)
    {
        char coinName[8];
        sprintf(coinName, "dragon%i", i+1);
        progress.coins[i] = Save::FlagIsSet(coinName);
        if (progress.coins[i])
        {
            coinCount++;
        }
    }
    u8 missionCount = 0;
    const Mission * mission;
    for (u8 i=0; i<GameProgress_MISSION_COUNT; i++)
    {
        mission = &missions[i];
        progress.missions[i] = Save::FlagIsSet(mission->hash);
        if (progress.missions[i])
        {
            missionCount++;
        }
    }
    progress.percent = 100 * ((missionCount + coinCount) * 100) / ((GameProgress_MISSION_COUNT + GameProgress_COIN_COUNT) * 100);
    progress.deaths = Save::GetDeaths();
    return progress;
}