#include "MapSequence.h"
#include <stdio.h>
#include <MapData.h>
#include <TileMap.h>
#include <loaders/TiledUtils.h>
#include <ui/GameUI.h>
#include <Actor.h>
#include <Save.h>
#include <Missions.h>
#include <Party.h>
#include <Scene.h>
#include <Player.h>
#include <MusicPlayer.h>
#include <behaviours/BlockBehaviour.h>
#include <behaviours/KillBehaviour.h>
#include <behaviours/TakeBehaviour.h>
#include <behaviours/TalkBehaviour.h>
#include <behaviours/WalkBehaviour.h>
#include <SwordGame.h>

MapSequence::MapSequence(TileMap * map, const MapObject * obj) : 
    Sequence(),
    map(map),
    host(obj)
{
    actorObj = MapEntityTypes::IsActor(host) ? host : NULL;
}

void MapSequence::Run()
{
    stackSize = 0;
    gameBeaten = false;
    state = SequenceState_RUNNING;
    Log_Info("Sequence start: %i\n", host->id);
    state = RunNext(host);
}

SequenceState MapSequence::RunNext(const MapObject * obj)
{
    if (gameBeaten)
        return SequenceState_COMPLETE;

    stackSize++;
    if (stackSize == UINT8_MAX)
    {
        Log_Error("Command stack overflow.\n");
        return SequenceState_ABORTED;
    }

    if (state >= SequenceState_COMPLETE)
        return state;

    if (obj != currentObj)
    {
        currentObj = obj;
        cmdIndex = 0;
    }

    if (obj->objType == MapObjectType_Entity)
    {
        actorObj = obj;
    }
    
    Log_Info("Sequence cmd: %i::%i\n", obj->id, cmdIndex);

    const MapCmd * cmd = NULL;
    const MapObject * next = NULL;
    while (cmdIndex < obj->cmdCount)
    {
        cmd = &map->data.cmds[obj->cmdFirst + cmdIndex];
        cmdIndex++;
        switch (cmd->type)
        {
            case MapCmd_IF:
                if (CheckIf(cmd, true))
                    next = GetNext(cmd);
                break;

            case MapCmd_IF_MONEY:
            {
                Player * player = GetActivePlayer();
                if (player->inventory.chicken >= cmd->param)
                    next = GetNext(cmd);
                break;
            }

            case MapCmd_IFNOT:
                if (CheckIf(cmd, false))
                    next = GetNext(cmd);
                break;

            case MapCmd_THEN:
                next = GetNext(cmd);
                break;

            case MapCmd_SAVE:
            {
                bool setSaveOn = cmd->type == MapCmd_SAVE;
                const char * saveName = map->data.names[cmd->value].Text();
                MissionState missionState = Missions::Inst().GetState(saveName);
                if (Save::FlagIsSet(saveName) == false)
                {
                    Save::FlagSet(saveName, setSaveOn);
                    if (MusicPlayer::Inst().HasCue(saveName))
                    {
                        MusicPlayer::Inst().Play(saveName);
                    }
                }
                map->UpdateFlags();
                if (HashEqual(saveName, "ending"))
                {
                    gameBeaten = true;
                }
                if (missionState == MissionState_INCOMPLETE)
                {
                    Missions::Inst().Complete(saveName);
                    if (gameBeaten)
                    {
                        MusicPlayer::Inst().Stop();
                    }
                    return SequenceState_PAUSED;
                }
                else if (gameBeaten)
                {
                    GameUI::Current()->SetMessage("You got him again!", 3.0f, MessageType_Standard);
                    MusicPlayer::Inst().Stop();
                    return SequenceState_PAUSED;
                }
                break;
            }

            case MapCmd_ACTOR:
            {
                const MapObject * newActorObj = map->GetEntityById(cmd->value);
                if (MapEntityTypes::IsActor(newActorObj))
                {
                    actorObj = newActorObj;
                }
                break;
            }

            case MapCmd_PARTY:
            {
                if (MapEntityTypes::IsActor(actorObj))
                {
                    Party::Inst().AddPartyMember((NPCActor*)actorObj->entity);
                }
                break;
            }

            case MapCmd_UNPARTY:
            {
                const MapObject * unpartyObj = map->GetEntityById(cmd->value);
                if (MapEntityTypes::IsActor(unpartyObj))
                {
                    MapEntityType type = MapEntityTypes::ToType(unpartyObj->type);
                    Party::Inst().RemovePartyMember(type);
                }
                break;
            }

            case MapCmd_KILL:
            {
                if (MapEntityTypes::IsActor(actorObj))
                {
                    Actor * victim = NULL;
                    if (cmd->value == 0)
                    {
                        victim = GetActivePlayer();
                    }
                    else
                    {
                        const MapObject * victimObj = map->GetEntityById(cmd->value);
                        if (MapEntityTypes::IsActor(victimObj))
                        {
                            victim = (Actor*)victimObj->entity;
                        }
                    }
                    if (victim != NULL)
                    {
                        Actor * killer = (Actor*)actorObj->entity;
                        waitForBehaviour = new KillBehaviour(killer, victim);
                        killer->SetBehaviour(waitForBehaviour);
                    }
                }
                return SequenceState_RUNNING;
            }

            case MapCmd_DEAD:
            {
                if (MapEntityTypes::IsActor(actorObj))
                {
                    ((Actor*)actorObj->entity)->Die();
                }
                break;
            }

            case MapCmd_FACTION:
            {
                if (MapEntityTypes::IsActor(actorObj))
                {
                    ((Actor*)actorObj->entity)->SetFaction(cmd->value);
                }
                break;
            }

            case MapCmd_TAKE:
            {
                Player * player = GetActivePlayer();
                if (player != NULL)
                {
                    Actor * taker = NULL;
                    const MapObject * takerObj = map->GetEntityById(cmd->value);
                    if (MapEntityTypes::IsActor(takerObj))
                    {
                        taker = (Actor*)takerObj->entity;
                        waitForBehaviour = new TakeBehaviour(taker, player);
                        taker->SetBehaviour(waitForBehaviour);
                        return SequenceState_RUNNING;
                    }
                }
                break;
            }

            case MapCmd_TAKE_MONEY:
            {
                Player * player = GetActivePlayer();
                if (player != NULL)
                {
                    player->inventory.chicken -= cmd->value;
                }
                break;
            }

            case MapCmd_WALK:
            {
                if (MapEntityTypes::IsActor(actorObj))
                {
                    const MapObject * walkTarget = map->GetObjectById(cmd->value);
                    if (walkTarget != NULL)
                    {
                        Actor * walker = (Actor*)actorObj->entity;
                        waitForBehaviour = new WalkBehaviour(walker, walkTarget);
                        walker->SetBehaviour(waitForBehaviour);
                        return SequenceState_RUNNING;
                    }
                }
                break;
            }

            case MapCmd_BLOCK:
            {
                if (MapEntityTypes::IsActor(actorObj))
                {
                    Actor * blocker = (Actor*)actorObj->entity;
                    blocker->SetBehaviour(new BlockBehaviour(blocker));
                }
                break;
            }

            case MapCmd_GIVE:
            {
                if (actorObj != NULL)
                {
                    Player * player = GetActivePlayer();
                    if (player != NULL)
                    {
                        player->SetHoldItemAnnounced(MapEntityTypes::ToType(cmd->value));
                        return SequenceState_PAUSED;
                    }
                }
                break;
            }

            case MapCmd_HOLD:
            {
                if (MapEntityTypes::IsActor(actorObj))
                {
                    Actor * holder = (Actor*)actorObj->entity;
                    holder->SetHoldItem(MapEntityTypes::ToType(cmd->value));
                }
                break;
            }

            case MapCmd_MUSIC:
            {
                const MapName * music = &map->data.names[cmd->value];
                MusicPlayer::Inst().Play(music->Text());
                break;
            }

            case MapCmd_SAY:
            {
                if (DoSay(obj, cmd, MessageType_Standard))
                {
                    if (cmdIndex < obj->cmdCount)
                        return SequenceState_PAUSED;
                }
                break;
            }

            case MapCmd_PROMPT:
            {
                if (DoSay(obj, cmd, MessageType_Confirm))
                {
                    return SequenceState_PAUSED;
                }
                break;
            }

            case MapCmd_HIDE:
            {
                if (actorObj->entity != NULL)
                {
                    actorObj->entity->SetEnabled(cmd->value != 1);
                }
                break;
            }
        }
        if (next != NULL)
        {
            return RunNext(next);
        }
    }
    state = SequenceState_COMPLETE;
    Stop();
    return state;
}

bool MapSequence::DoSay(const MapObject * obj, const MapCmd * cmd, MessageType messageType)
{
    GameUI * ui = GameUI::Current();
    if (ui == NULL)
    {
        Log_Error("Couldn't get UI on %s\n", obj->name.Text());
        return false;
    }
    const MapMessage * msg = &map->data.messages[cmd->value];
    const char * msgStr = msg->Text();
    if (msgStr == NULL || msgStr[0] == '\0')
    {
        Log_Error("Couldn't find message %i on %s\n", cmd->value, obj->name.Text());
        return false;
    }
    if (actorObj != NULL && actorObj->objType == MapObjectType_Entity && MapStr::IsValid(&actorObj->name))
    {
        ui->SetTalkMessage(MapStr::ReadText(&actorObj->name), msgStr, 3.0f, messageType);
    }
    else
    {
        ui->SetMessage(msgStr, 3.0f, messageType);
    }
    if (actorObj != NULL && actorObj->entity != NULL)
    {
        if (MapEntityTypes::IsActor(actorObj))
        {
            Actor * talkActor = (Actor*)actorObj->entity;
            Player * player = GetActivePlayer();
            if (player != NULL)
            {
                waitForBehaviour = new TalkBehaviour(talkActor, player);
                talkActor->SetBehaviour(waitForBehaviour);
            }
        }
    }
    return true;
}

bool MapSequence::CheckIf(const MapCmd * cmd, bool positiveValue)
{
    return map->FlagIsSet(cmd->param) == positiveValue;
}

const MapObject * MapSequence::GetNext(const MapCmd * cmd)
{
    const MapObject * nextObj = map->GetArea(cmd->value);
    if (nextObj == NULL)
    {
        Log_Error("Couldn't find next object %i from %s\n", cmd->value, currentObj->name.Text());
        return NULL;
    }
    return nextObj;
}

const MapObject * MapSequence::GetCurrentObj() const
{
    if (currentObj == NULL || state >= SequenceState_COMPLETE)
        return NULL;
    return currentObj;
}

const MapCmd * MapSequence::GetCurrentCmd() const
{
    if (currentObj == NULL || state >= SequenceState_COMPLETE)
        return NULL;
    if (cmdIndex - 1 >= currentObj->cmdCount)
        return NULL;
    return &map->data.cmds[currentObj->cmdFirst + cmdIndex - 1];
}

void MapSequence::Resume()
{
    if (gameBeaten)
    {
        Save::SetMapCurrent(0);
        Save::SetSaveMap(0);
        Save::SaveFile();
        SwordGame::Inst().GameWin();
        return;
    }
    waitForBehaviour = NULL;

    const MapCmd * prevCmd = GetCurrentCmd();
    if (prevCmd != NULL && prevCmd->type == MapCmd_PROMPT)
    {
        if (GameUI::Current()->GetMessageStatus() == MessageStatus_Confirm)
        {
            const MapObject * confirmNext = map->GetArea(prevCmd->param);
            state = RunNext(confirmNext);
            return;
        }
    }
    
    if (cmdIndex >= currentObj->cmdCount)
    {
        state = SequenceState_COMPLETE;
    }
    else
    {
        Log_Info("Sequence resume: %i\n", currentObj->id);
        state = RunNext(currentObj);
    }
}

bool MapSequence::CanContinue()
{
    if (state != SequenceState_RUNNING)
        return false;

    if (waitForBehaviour == NULL)
        return true;

    bool isDone = waitForBehaviour->IsDone();
    if (isDone)
    {
        return true;
    }
    return false;
}

Entity * MapSequence::GetTarget() const
{
    if (actorObj != NULL)
        return actorObj->entity;
    return NULL;
}

Player * MapSequence::GetActivePlayer()
{
    Player * player = (Player*)map->scene->GetNearestEntity(Vector3_ZERO, ActorFaction_PLAYER, 1000.0f);
    if (player == NULL || player->IsEnabled() == false || player->inventory.hearts == 0)
        return NULL;
    return player;
}

void MapSequence::Stop()
{
    if (state < SequenceState_COMPLETE)
    {
        state = SequenceState_ABORTED;
    }
}
