#include "MapEnvironment.h"
#include <read/tinyxml2.h>
#include <loaders/TiledUtils.h>
#include <FileSystem.h>

using namespace tinyxml2;

std::vector<MapEnvInfo> MapEnvironment::envs;
std::vector<MapLayerInfo> MapEnvironment::layers;
bool MapEnvironment::hasLoaded = false;

const MapEnvInfo * MapEnvironment::GetEnvInfo(HashInt hash)
{
    Load();
    for (std::vector<MapEnvInfo>::iterator it = envs.begin(); it != envs.end(); ++it)
    {
        if (it->hash == hash)
            return &(*it);
    }
    return NULL;
}

const MapEnvInfo * MapEnvironment::GetEnvInfo(const char * name)
{
    return GetEnvInfo(Hash(name));
}

const MapLayerInfo * MapEnvironment::GetLayerInfo(HashInt hash)
{
    Load();
    for (std::vector<MapLayerInfo>::iterator it = layers.begin(); it != layers.end(); ++it)
    {
        if (it->hash == hash)
            return &(*it);
    }
    return NULL;
}

const MapLayerInfo * MapEnvironment::GetLayerInfo(const char * name)
{
    return GetLayerInfo(Hash(name));
}

void MapEnvironment::Load()
{
    if (hasLoaded)
        return;

    hasLoaded = true;

    envs.clear();
    layers.clear();

    XMLDocument doc;
    FileSystem::LoadXML("data/environment.xml", &doc);
    if (doc.Error())
    {
        Log_Error("Couldn't load environement");
        return;
    }

    XMLElement * rootNode = doc.FirstChildElement("environment");
    if (rootNode == NULL)
    {
        Log_Error("Couldn't find valid environment in file.");
        return;
    }

    XMLElement * layersNode = rootNode->FirstChildElement("layers");
    if (layersNode == NULL)
    {
        Log_Error("Couldn't find valid layers in file.");
        return;
    }

    XMLElement * envsNode = rootNode->FirstChildElement("envs");
    if (envsNode == NULL)
    {
        Log_Error("Couldn't find valid envs in file.");
        return;
    }

    XMLElement * layerNode = layersNode->FirstChildElement("layer");
    while (layerNode != NULL)
    {
        const char * id = layerNode->Attribute("id");
        layers.push_back(MapLayerInfo());
        MapLayerInfo & layer = layers.back();
        layer.hash = Hash(id);
        layer.textureSurface = MapPath(layerNode->Attribute("surf"));
        layer.textureEdge = MapPath(layerNode->Attribute("edge"));
        layer.textureTree = MapPath(layerNode->Attribute("tree"));
        layer.shader = MapPath(layerNode->Attribute("shader"));
        layer.slip = layerNode->FloatAttribute("slip", 0.0f);
        layer.damage = layerNode->UnsignedAttribute("damage", 0);
        layerNode = layerNode->NextSiblingElement();
    }

    XMLElement * envNode = envsNode->FirstChildElement("env");
    char att[8];
    const char * layerId;
    HashInt layerHash;
    const MapLayerInfo * layer;
    while (envNode != NULL)
    {
        const char * id = envNode->Attribute("id");
        envs.push_back(MapEnvInfo());
        MapEnvInfo & env = envs.back();

        env.hash = Hash(id);
        env.border = envNode->UnsignedAttribute("border");
        strcpy(env.music, "\0");
        const char * music = envNode->Attribute("music");
        if (music != NULL)
        {
            strncpy(env.music, music, MAP_NAME_LENGTH);
        }
        for (u8 i=0; i<MAP_LAYER_COUNT; i++)
        {
            sprintf(att, "layer%i\0", i + 1);
            layerId = envNode->Attribute(att);
            layer = NULL;
            layerHash = Hash(layerId);
            if (layerHash > 0)
            {
                for (std::vector<MapLayerInfo>::iterator it = layers.begin(); it != layers.end(); ++it)
                {
                    if (it->hash == layerHash)
                    {
                        layer = &(*it);
                        break;
                    }
                }
            }
            env.layers[i] = layer;
        }

        // do water layer
        layer = NULL;
        HashInt waterLayerHash = Hash(envNode->Attribute("water"));
        if (waterLayerHash == 0)
        {
            waterLayerHash = Hash("water");
        }
        for (std::vector<MapLayerInfo>::iterator it = layers.begin(); it != layers.end(); ++it)
        {
            if (it->hash == waterLayerHash)
            {
                layer = &(*it);
                break;
            }
        }
        env.layers[MAP_LAYER_WATER] = layer;

        envNode = envNode->NextSiblingElement();
    }
}


void MapEnvironment::MarkUnloaded()
{
    hasLoaded = false;
}
