#include <MapEntity.h>
#include <Maths.h>
#include <Collider.h>
#include <TileMap.h>
#include <MapEnvironment.h>
#include <CollisionChecker.h>

MapEntity::MapEntity(MapEntityType _type)
    : type(_type)
{
    info = &MapObject::BLANK;
}

MapEntity::~MapEntity()
{
}

void MapEntity::Reset()
{
    velocity = Vector3_ZERO;
    moveVelocity = Vector3_ZERO;
    slipVelocity = Vector3_ZERO;
    dirCollision = Vector3_ZERO;
    fall = 0.0f;
    knockTime = 0.0f;
}

void MapEntity::Added()
{
    const MapCmd * cmd = map->GetObjectCmd(info, MapCmd_HIDE);
    if (cmd != NULL)
    {
        SetEnabled(cmd->value != 1);
    }
}

void MapEntity::Start()
{
    if (collider != NULL && collider->mode == CollisionMode_Init)
    {
        SetCollision(CollisionMode_None);
    }
    if (hasSpawned == false)
    {
        Spawn(transform.position);
    }
}

void MapEntity::Spawn(Vector3 position)
{
    spawnPos = position;
    spawnRot = transform.rotation;
    Respawn();
    hasSpawned = true;
}

void MapEntity::Respawn()
{
    Reset();
    transform.SetPosition(spawnPos).SetRotation(spawnRot);
}

void MapEntity::Update(float dt)
{
    if (knockTime > 0.0f)
    {
        moveVelocity = dirMove * -knockPower;
        if (knockPower > 0.0f)
        {
            knockPower -= dt * 60.0f;
            if (knockPower < 0.0f)
            {
                knockPower = 0.0f;
            }
        }
        knockTime -= dt;
        if (knockTime <= 0.0f)
        {
            knockTime = 0.0f;
        }
        isMoving = true;
    }
    else
    {
        Vector3 moveVelocityTo = GetMoveVelocity(dt);
        moveVelocity = Maths::Lerp(
            moveVelocity, 
            moveVelocityTo, 
            Maths::Clamp01(Maths::LengthSq(moveVelocityTo) > Maths::LengthSq(moveVelocity) ? acceleration : deceleration), 
            dt
        );
        isMoving = Maths::LengthSq(moveVelocity) > 0.01f;
        if (Maths::LengthSq(moveVelocityTo) > 0.01f)
        {
            dirMove = Maths::Normalize(moveVelocityTo);
        }
    }

    float slip = layerInfo != NULL ? layerInfo->slip : 0.0f;
    if (map != NULL && Maths::LengthSq(moveVelocity + slipVelocity) > 0.001f)
    {
        // wall detection
        Vector3 wantDirMove = Maths::Normalize(moveVelocity + slipVelocity);
        Vector3 wallPos = transform.position + wantDirMove * radius;
        float wallY = map->GetHeight(wallPos);
        float faceDot = Maths::Dot(dirMove, wantDirMove);
        if (wallY > transform.position.y + maxGradient/*&& faceDot >= 0.0f*/)
        {
            isMoving = false;

            // layer damage
            i16 wx, wz;
            map->GetTile(wallPos, &wx, &wz);
            const MapLayerInfo * layer = map->GetLayerInfo(wx, wz);
            if (layer != NULL && layer->damage > 0)
            {
                TakeHit(NULL, layer->damage);
            }
        }
        if (isMoving == false)
        {
            moveVelocity = Vector3_ZERO;
            //slipVelocity = Vector3_ZERO;
            knockPower = 0.0f;
        }
        // recent collision detection
        if (Maths::LengthSq(dirCollision) > 0.1f)
        {
            if (Maths::Dot(wantDirMove, dirCollision) > 0.0f)
            {
                if (Maths::Abs(dirCollision.x) > Maths::Abs(dirCollision.z))
                {
                    moveVelocity.x = 0.0f;
                }
                else
                {
                    moveVelocity.z = 0.0f;
                }
            }
            isMoving = Maths::LengthSq(moveVelocity) > 0.01f;
        }
        if (isMoving && knockTime <= 0.0f)
        {
            if (transform.position.y > wallY + 0.1f)
            {
                slip = 0.0f;
            }
        }
        if (Maths::LengthSq(slipVelocity) > 0.1f)
        {
            wallPos = transform.position + Maths::Normalize(slipVelocity) * radius;
            wallY = map->GetHeight(wallPos);
            if (wallY > transform.position.y + maxGradient)
            {
                slipVelocity = Vector3_ZERO;
            }
        }
    }

    if (slip > 0.0f)
    {
        slipVelocity += moveVelocity * dt * 2.0f;
        slipVelocity = Maths::Clamp(slipVelocity, 5.0f);
        moveVelocity *= 0.3f;
    }
    else
    {
        slipVelocity = Vector3(0.0f);
    }

    velocity = moveVelocity;
    velocity += slipVelocity;
    velocity.y = -fall;
    
    isGrounded = true;
    
    Vector3 positionTo = transform.position + velocity * dt;
    if (map != NULL && useGravity)
    {
        groundY = map->GetHeight(transform.position);

        map->GetTile(transform.position, &point.x, &point.z);
        mapTile = map->GetTileInfo(point.x, point.z);
        layer = mapTile.layer;
        layerInfo = map->GetLayerInfo(layer);
        if (layer == MAP_LAYER_WATER + 1)
        {
            groundY = waterHeight;
        }
        else if (groundY <= -1.0f)
        {
            groundY = MapEntity_KNOCKOUT_Y;
        }

        float groundDiff = groundY - transform.position.y;
        if (groundDiff < 0.0f)
        {
            if (fall < 20.0f)
            {
                fall += dt * 60.0f;
            }
            isGrounded = false;
        }
        else if (fall >= 0.0f)
        {
            fall = 0.0f;
            positionTo.y = groundY;
        }

        if (positionTo.y < groundY)
        {
            positionTo.y = groundY;
        }
    }

    transform.position = positionTo;

    UpdateAnimation(dt);
    
    if (transform.position.y <= -10.0f)
    {
        OutOfMap();
    }

    dirCollision = Vector3_ZERO;
}

void MapEntity::SetCollision(CollisionMode mode, Vector3 size, Vector3 center)
{
    if (collider == NULL)
    {
        collider = new Collider(this, size, center);
    }
    collider->mode = mode;
    if (Maths::LengthSq(size) < 0.01f)
    {
        size = Vector3(radius * 2.0f);
        center = Vector3(0, radius, 0);
    }
    collider->size = size;
    collider->center = center;
}

Vector3 MapEntity::GetMoveVelocity(float dt)
{
    return Vector3_ZERO;
}

bool MapEntity::TakeHit(Entity * other, u8 power)
{
    Knock(other);
    return true;
}

void MapEntity::Knock(Entity * other)
{
    if (knockTime > 0.0f)
        return;
    Vector3 dir = other == NULL ? -transform.Forwards() : transform.position - other->transform.position;
    dir.y = 0.0f;
    dir = Maths::Normalize(dir);
    Knock(dir);
}

void MapEntity::Knock(Vector3 dir)
{
    if (knockTime > 0.0f)
        return;
    dirMove = -dir;
    knockTime = MapEntity_KNOCK_TIME;
    knockPower = 15.0f;
}

void MapEntity::OnCollision(Entity * other)
{
    fall = 0.0f;
    if (other != NULL && other->collider->mode >= CollisionMode_Collide && other->collider->mass > 0.0f)
    {
        dirCollision = other->transform.position - transform.position;
        if (dirCollision.y - other->collider->center.y < maxGradient * 2.0f)
        {
            dirCollision = Vector3_ZERO;
        }
        else if (dirCollision.y < Maths::Abs(dirCollision.x) + Maths::Abs(dirCollision.z))
        {
            dirCollision.y = 0.0f;
            Maths::Normalize(dirCollision);
        }
        else
        {
            dirCollision = Vector3_ZERO;
        }
    }
}

void MapEntity::OutOfMap()
{
    if (canDespawn)
        SetEnabled(false);
}