#ifndef _DESS_MAP_DATA_H
#define _DESS_MAP_DATA_H

#include <desstypes.h>
#include <Direction.h>
#include <Hash.h>
#include <string>

#define DEFAULT_TILE_SIZE 2.0f
#define DEFAULT_TILE_HEIGHT 0.75f
#define MAP_NAME_LENGTH 16
#define MAP_PATH_LENGTH 64
#define MAP_MESSAGE_LENGTH 256
#define MAP_LAYER_COUNT 7
#define MAP_LAYER_WATER (MAP_LAYER_COUNT - 1)
#define MAP_FLAG_COUNT 32
#define DEFAULT_BORDER 2
#define MAP_WATER_HEIGHT 0.3f

class Entity;

enum TileType
{
    TileType_Empty,
    TileType_Solid,
    TileType_CornerNE,
    TileType_CornerSE,
    TileType_CornerSW,
    TileType_CornerNW,
    TileType_RampN,
    TileType_RampE,
    TileType_RampS,
    TileType_RampW
};

enum LayerType
{
    LayerType_Ground,
    LayerType_Wall
};

struct TileInfo
{
    TileType type = TileType_Empty;
    u8 layer = 0;
    u8 under = 0;
    TileInfo(TileType type = TileType_Empty, u8 layer = 0, u8 under = 0) : type(type), layer(layer), under(under) {}
};

enum MapObjectType
{
    MapObjectType_None,
    MapObjectType_MapArea,
    MapObjectType_Trigger,
    MapObjectType_TriggerUsed,
    MapObjectType_MessageArea,
    MapObjectType_Load,
    MapObjectType_Entity,
    MapObjectType_SpawnPoint,
    MapObjectType_Model
};

class MapStr
{
private:
    u16 length = 0;
    char * text = NULL;
public:
    static const char* ReadText(const MapStr* str, const char* defaultValue = NULL);
    static bool IsValid(const MapStr * str);

    MapStr(const char * src = NULL, u16 length = 0);
    MapStr(const MapStr & original);
    virtual ~MapStr();

    MapStr & operator=(const MapStr& other);

    inline u16 Length() const { return length; }
    const char * Text() const { return length > 0 ? text : NULL; }
};

struct MapName : public MapStr
{
    MapName(const char * src = NULL) : MapStr(src, MAP_NAME_LENGTH) {}
};

struct MapPath : public MapStr
{
    MapPath(const char * src = NULL) : MapStr(src, MAP_PATH_LENGTH) {}
};

struct MapMessage : public MapStr
{ 
    MapMessage(const char * src = NULL) : MapStr(src, MAP_MESSAGE_LENGTH) {}
};

enum MapCmdType
{
    MapCmd_NONE,
    MapCmd_IF,
    MapCmd_IF_MONEY,
    MapCmd_IFNOT,
    MapCmd_MAP,
    MapCmd_ACTOR,
    MapCmd_UNLOCK,
    MapCmd_GIVE,
    MapCmd_FLAG,
    MapCmd_SAVE,
    MapCmd_PARTY,
    MapCmd_UNPARTY,
    MapCmd_TAKE,
    MapCmd_TAKE_MONEY,
    MapCmd_KILL,
    MapCmd_WALK,
    MapCmd_SAY,
    MapCmd_PROMPT,
    MapCmd_DEAD,
    MapCmd_BLOCK,
    MapCmd_FACTION,
    MapCmd_HOLD,    
    MapCmd_MUSIC,  
    MapCmd_DIR,
    MapCmd_MODEL,
    MapCmd_TEXTURE,
    MapCmd_ENTER,
    MapCmd_HIDE,
    MapCmd_THEN
};

struct MapCmd
{
    MapCmdType type = MapCmd_NONE;
    u8 param = 0;
    u8 value = 0;
};

struct MapPoint 
{
    i16 x = 0;
    i16 z = 0;
    bool operator==(const MapPoint &o) const { return x == o.x && z == o.z; }
    bool operator!=(const MapPoint &o) const { return x != o.x || z != o.z; }
    MapPoint(i16 x = 0, i16 z = 0) : x(x), z(z) {}
};

const MapPoint MapPoint_MIN(INT16_MIN, INT16_MIN);

struct MapObject
{
    MapObjectType objType = MapObjectType_None;
    u8 id = 0;
    u8 type = 0;
    i16 x = 0;
    i16 z = 0;
    u16 width = 1;
    u16 depth = 1;
    u8 scale = 1;
    u8 rotation = 0;
    bool flipH = false;
    bool flipV = false;
    bool flipD = false;
    u16 flag = 0;
    MapName name;
    HashInt nameHash = 0;
    MapMessage * message = NULL;
    Entity * entity = NULL;
    u8 cmdFirst = 0;
    u8 cmdCount = 0;
    
    inline bool Contains(i16 px, i16 pz) const
    {
        return px >= x && pz >= z && px <= x + width && pz <= z + depth;
    }

    void Constrain(i16 & px, i16 & pz) const;

    static MapObject BLANK;
};

struct MapData
{
    u8 width = 0;
    u8 depth = 0;
    u8 pixelsPerTile = 16;
    u8 border = UINT8_MAX;
    float tileSize = DEFAULT_TILE_SIZE;
    float tileHeight = DEFAULT_TILE_HEIGHT;
    float zoom = 1.0f;
    HashInt envHash = 0;
    HashInt layerHashes[MAP_LAYER_COUNT];
    char name[MAP_NAME_LENGTH];
    std::vector<TileInfo> tiles;
    std::vector<MapObject> areas;
    std::vector<MapObject> entities;
    std::vector<MapName> names;
    std::vector<MapMessage> messages;
    std::vector<MapCmd> cmds;
    MapPath music;

    MapData() : music(NULL) {}

    void Clear()
    {
        width = 0;
        depth = 0;
        pixelsPerTile = 16;
        border = UINT8_MAX;
        tileSize = DEFAULT_TILE_SIZE;
        tileHeight = DEFAULT_TILE_HEIGHT;
        envHash = 0;
        for (u8 i=0; i<MAP_LAYER_COUNT; i++)
        {
            layerHashes[i] = 0;
        }
        name[0] = '\0';
        tiles.clear();
        areas.clear();
        entities.clear();
        names.clear();
        messages.clear();
        cmds.clear();
        zoom = 1.0f;
        music = NULL;
    }
};

class MapUtils
{
public:
    static std::string MapPathToName(const char * mapPath);
    static HashInt MapPathToHash(const char * mapPath);
    static std::string MapHashToPath(HashInt hash);
};

enum MapLoadFlag
{
    MapLoadFlag_Default,
    MapLoadFlag_NoCommands = 0x1,
    MapLoadFlag_NoActors = 0x2
};

class MapLoader
{
public:
    virtual bool CanLoad(const char * path) = 0;
    virtual bool Load(const char * path, MapData * out, u16 loadFlags = 0) = 0;
};

#endif