#ifndef _ACTOR_BEHAVIOUR_H
#define _ACTOR_BEHAVIOUR_H

#include <desstypes.h>
#include <Actor.h>
#include <Maths.h>

enum BehaviourRunState
{
    BehaviourRunState_INIT,
    BehaviourRunState_RUNNING,
    BehaviourRunState_DONE
};

const Vector3 INVALID_POS(INFINITY);

class ActorBehaviour
{
protected:
    Actor * actor = NULL;
    Actor * target = NULL;
    Vector3 targetPos = INVALID_POS;
    float speed = 0.0f;
    float stateTime = 0.0f;
    float nextStateTime = 0.0f;
    u8 state = 0;
    BehaviourRunState runState = BehaviourRunState_INIT;

public:
    bool avoidEdges = false;
    float minDot = 0.0f;

    ActorBehaviour(Actor * actor) 
        : actor(actor)
    {
        speed = actor->GetMoveSpeed();
        targetPos = INVALID_POS;
    }

    virtual void Start() 
    {
        speed = actor->GetMoveSpeed();
        state = PickState();
        runState = BehaviourRunState_RUNNING;
    }
    
    void SetTarget(Actor * value)
    {
        target = value;
    }
    
    virtual void Update(float dt)
    {
        if (actor->IsEnabled() == false || actor->scene == NULL)
        {
            state = 0;
            stateTime = 0.0f;
            return;
        }

        u8 prevState = state;
        float prevNextStateTime = nextStateTime;
        state = UpdateState(dt);
        if (nextStateTime > 0.0f)
        {
            nextStateTime -= dt;
            if (nextStateTime < 0.0f)
            {
                state = 0;
            }
        }
        if (state == 0)
        {
            nextStateTime = 0.0f;
            state = PickState();
        }
        if (state != prevState)
        {
            stateTime = 0.0f;
            if (prevNextStateTime == nextStateTime + dt)
            {
                nextStateTime = 0.0f;
            }
        }
        else
        {
            stateTime += dt;
        }
    }

    virtual u8 PickState() { return 0; }

    virtual u8 UpdateState(float dt) { return 0; }

    virtual Vector3 GetMoveVelocity(float dt)
    {
        if (speed == 0.0f || actor->IsEnabled() == false)
            return Vector3_ZERO;
        if (targetPos == INVALID_POS)
            return Vector3_ZERO;
        Vector3 diff = targetPos - actor->transform.position;
        diff.y = 0.0f;
        if (Maths::LengthSq(diff) < speed * dt)
            return Vector3_ZERO;
        if (avoidEdges && actor->map != NULL)
        {
            float groundY = actor->map->GetHeight(actor->transform.position + Maths::Normalize(diff));
            if (groundY < actor->transform.position.y - 0.5f)
                return Vector3_ZERO;
        }
        return Maths::Clamp(diff * 40.0f, speed);
    }

    virtual void Chill() {}

    virtual bool TakeHit() { return false; }

    virtual void OnCollision(Entity * other) {}

    virtual void Stop() 
    {
        Done();
    }

    virtual float GetPickWeight() { return 1.0f; }

    inline bool IsDone() { return runState >= BehaviourRunState_DONE; }
    inline bool IsRunning() { return runState == BehaviourRunState_RUNNING; }

    void Done()
    {
        target = NULL;
        runState = BehaviourRunState_DONE;
    }

    friend class Actor;

};

#endif