#ifndef _ACTOR_H
#define _ACTOR_H

#include <desstypes.h>
#include <MapEntity.h>
#include <Model3D.h>
#include <MeshRenderer.h>
#include <Mesh.h>

#define ACTOR_SHADER_PATH "shaders/actor.dsh"

class TileMap;
struct MapObject;
class ActorBehaviour;

struct ActorInventory
{
    u8 hearts = 1;
    u8 maxHearts = 4;
    u8 chicken = 0;
    MapEntityType holdItem = MapEntityType_None;
};

enum ActorFaction
{
    ActorFaction_NONE = 0x0,
    ActorFaction_PLAYER = 0x1,
    ActorFaction_GOOD = 0x2,
    ActorFaction_EVIL = 0x4
};

enum ActorAnim
{
    ActorAnim_Idle,
    ActorAnim_Run,
    ActorAnim_Push,
    ActorAnim_StrikeFore,
    ActorAnim_StrikeBack,
    ActorAnim_Talk,
    ActorAnim_Jump,
    ActorAnim_Air,
    ActorAnim_Land,
    ActorAnim_Hit,
    ActorAnim_Die,
    ActorAnim_Block,
    ActorAnim_Get,
    ActorAnim_Roll,
    ActorAnim_Pose,
    ActorAnim_COUNT
};

const MapPoint WalkPoint_NONE = MapPoint_MIN;

class Actor : public MapEntity
{
protected:
    Model3D model;
    MapEntity * holdItem = NULL;
    ModelAnimData * anims[ActorAnim_COUNT];
    const ModelBoneData * boneHold = NULL;
    float animDownTime = 0.0f;
    float animMoveMulti = 1.0f;
    float rollDownTime = 0.0f;
    bool wantsStrike = false;
    bool wantsRoll = false;
    float timeToSwipe = 0.0f;
    u8 strikeI = 0;
    MapPoint walkPoint = WalkPoint_NONE;
    float timeInvincible = 0.0f;
    float timeToDie = 0.0f;
    std::vector<ActorBehaviour*> behaviours;
    ActorBehaviour * behaviour = NULL;
    float behaviourTime = 0.0f;
    u8 strikeCount = 2;
    float moveSpeed = 6.0f;
    float animPauseTime = 0.0f;
    bool permadeath = false;
    u8 initHearts = 1;
    Actor * attacker = NULL;

public:
    static bool IsActor(const Entity * entity);
    static bool IsActive(const Entity * entity);

    ActorInventory inventory;
    bool removeOnDeath = true;

    Actor(MapEntityType type);
    virtual ~Actor();

    virtual void Added();
    virtual void Start();
    virtual void LoadModel();
    virtual void Reset();

    virtual void Update(float dt);

    virtual void SetHoldItem(MapEntityType entityType);

    virtual void Strike();
    virtual void OnStrike(MapEntity * other, u8 power);

    virtual void Roll();

    void SetBehaviour(ActorBehaviour * behavior, bool override = true);

    void PlayAnimation(ActorAnim animation, float duration = 1.0f, float fps = 30.0f);
    void BlendAnimation(ActorAnim animation, float blendTime, float duration = 1.0f, float fps = 30.0f);
    void SetAnimation(ActorAnim animation, float time, float pauseTime = 0.0f);

    virtual bool TakeHit(Entity * other, u8 power);
    virtual void Chill();
    virtual void Die();
    void Face(Entity * other);

    Actor* FindTarget() const;
    u8 GetOpposisingFactionMask() const;
    bool IsWithinRange(Vector3 targetPosition, float range, float minDot = 0.0f) const;
    bool IsWithinRange(Entity * target, float range, float minDot = 0.0f) const;

    virtual void OnCollision(Entity * other);

    inline float GetMoveSpeed() const { return moveSpeed; }

    void Render();

protected:
    virtual void UpdateAnimation(float dt);
    virtual Vector3 GetMoveVelocity(float dt);
    virtual void OnDead();
    virtual void OutOfMap();
    virtual void TakeSwipe();
};

#endif