#include <Actor.h>
#include <Camera.h>
#include <Maths.h>
#include <MapEntityFactory.h>
#include <fx/FXer.h>
#include <desslog.h>
#include <Scene.h>
#include <AttackManager.h>
#include <ActorBehaviour.h>
#include <Save.h>

// Animations: Idle,Run,PushIdle,PushWalk,StrikeFore,StrikeBack,Talk,Hit1,BlockVert,Jump,Air,Land,Die,Get,Roll,Pose

bool Actor::IsActor(const Entity * entity)
{
    return Entity::NotDestroyed(entity) && MapEntityTypes::IsActor(((Actor*)entity)->GetType());
}

bool Actor::IsActive(const Entity * entity)
{
    return Entity::IsActive(entity) && IsActor(entity);
}

Actor::Actor(MapEntityType type) : 
    MapEntity(type)
{
    dirMove = Vector3_SOUTH;
    transform.LookDirection(Vector3_SOUTH);
}

Actor::~Actor()
{
    AttackManager::Remove(this);
    for (size_t i=0; i<behaviours.size(); i++)
    {
        delete behaviours[i];
    }
    behaviours.clear();
}

void Actor::Added()
{
    MapEntity::Added();
    transform.LookDirection(Vector3_SOUTH);
}

void Actor::Reset()
{
    Chill();
    AttackManager::Remove(this);
    MapEntity::Reset();
    timeToDie = 0.0f;
    timeToSwipe = 0.0f;
    timeInvincible = 0.0f;
    animDownTime = 0.0f;
}

void Actor::Start()
{
    MapEntity::Start();

    LoadModel();

    strikeCount = 1;
    if (anims[ActorAnim_StrikeBack] != NULL && anims[ActorAnim_StrikeFore] != NULL)
    {
        strikeCount = 2;
    }

    if (info->cmdCount > 0)
    {
        for (u8 i = 0; i < info->cmdCount; i++)
        {
            MapCmd *cmd = &map->data.cmds[info->cmdFirst + i];
            if (cmd->type == MapCmd_DEAD && map->FlagIsSet(cmd->value))
            {
                Die();
            }
        }
    }

    if (inventory.hearts > 0)
    {
        model.PlayAnimation(anims[ActorAnim_Idle]);
        SetCollision(CollisionMode_Resolve);
    }
    
    if (permadeath && map->FlagIsSet(info->id))
    {
        Die();
    }

    initHearts = inventory.hearts;
    if (faction == ActorFaction_EVIL && Save::FlagIsSet(SAVE_FLAG_NEWGAMEPLUS))
    {
        inventory.hearts = initHearts * 2;
    }
}

void Actor::LoadModel()
{ 
    for (u8 i=0; i<ActorAnim_COUNT; i++)
    {
        anims[i] = NULL;
    }
    boneHold = NULL;

    if (model.IsLoaded() == false)
        return;

    model.fps = 80.0f;
    model.transform.rotation = glm::angleAxis(PI, Vector3_UP);
    anims[ActorAnim_Idle] = model.FindAnimation("Idle");
    if (anims[ActorAnim_Idle] != NULL)
        anims[ActorAnim_Idle]->loop = true;
    anims[ActorAnim_Run] = model.FindAnimation("Run");
    if (anims[ActorAnim_Run] != NULL)
        anims[ActorAnim_Run]->loop = true;
    anims[ActorAnim_Push] = model.FindAnimation("PushWalk");
    if (anims[ActorAnim_Push] != NULL)
        anims[ActorAnim_Push]->loop = true;
    anims[ActorAnim_StrikeFore] = model.FindAnimation("StrikeFore");
    anims[ActorAnim_StrikeBack] = model.FindAnimation("StrikeBack");
    anims[ActorAnim_Hit] = model.FindAnimation("Hit1");
    anims[ActorAnim_Die] = model.FindAnimation("Die");
    anims[ActorAnim_Block] = model.FindAnimation("BlockVert");
    anims[ActorAnim_Jump] = model.FindAnimation("Jump");
    anims[ActorAnim_Air] = model.FindAnimation("Air");
    anims[ActorAnim_Land] = model.FindAnimation("Land");
    anims[ActorAnim_Talk] = model.FindAnimation("Talk");
    if (anims[ActorAnim_Talk] != NULL)
        anims[ActorAnim_Talk]->loop = true;
    anims[ActorAnim_Get] = model.FindAnimation("Get");
    anims[ActorAnim_Roll] = model.FindAnimation("Roll");
    anims[ActorAnim_Pose] = model.FindAnimation("Pose");
    boneHold = model.FindBone("B_Weapon.L");
    if (boneHold == NULL)
    {
        boneHold = model.FindBone("B_Hand.L");
    }
}

void Actor::Update(float dt)
{
    MapEntity::Update(dt);
    behaviour = behaviours.size() > 0 ? behaviours.back() : NULL;
    if (behaviour != NULL && behaviour->IsDone())
    { 
        behaviours.pop_back();
        behaviour = behaviours.size() > 0 ? behaviours.back() : NULL;
        behaviourTime = 0.0f;
    }
    if (behaviour != NULL && inventory.hearts > 0)
    {
        if (behaviour->IsRunning() == false)
        {
            behaviour->Start();
            behaviourTime = 0.0f;
        }
        behaviour->Update(dt);
        behaviourTime += dt;
    }
}

void Actor::UpdateAnimation(float dt)
{
    if (timeToDie > 0.0f)
    {
        timeToDie -= dt;
        if (timeToDie < 0.0f)
        {
            timeToDie = 0.0f;
            OnDead();
        }
    }
    else if (animDownTime > 0.0f)
    {
        animDownTime -= dt;
    }
    else if (inventory.hearts > 0)
    {
        if (isMoving)
        {
            model.BlendAnimation(anims[ActorAnim_Run], 0.05f);
            float m = Maths::LengthSq(slipVelocity) > 0.01f ? 1600.0f : 400.0f;
            model.fps = Maths::Length(moveVelocity) * dt * m;
        }
        else
        {
            model.BlendAnimation(anims[ActorAnim_Idle], 0.1f);
            model.fps = 15.0f;
        }

        if (wantsStrike)
        {
            Strike();
        }
    }

    if (rollDownTime > 0.0f)
    {
        rollDownTime -= dt;
    }
    else if (wantsRoll)
    {
        Roll();
    }

    if (Maths::LengthSq(dirMove) > 0.3f)
    {
        Quat targetRot = Maths::LookRotation(dirMove);
        transform.rotation = Maths::Slerp(transform.rotation, targetRot, 0.3f, dt);
    }

    if (animPauseTime > 0.0f)
    {
        animPauseTime -= dt;
    }
    else
    {
        model.Update(dt);
    }

    if (timeToSwipe > 0.0f)
    {
        timeToSwipe -= dt;
        if (timeToSwipe <= 0.0f)
        {
            timeToSwipe = 0.0f;
            TakeSwipe();
        }
    }

    if (timeInvincible > 0.0f)
    {
        timeInvincible -= dt;
        if (timeInvincible < 0.0f)
        {
            timeInvincible = 0.0f;
        }
    }
}

void Actor::Strike()
{
    if (inventory.hearts == 0)
        return;
    model.PlayAnimation(anims[ActorAnim_StrikeFore + strikeI], 0.0, 30.0f);
    animDownTime = 0.35f;
    animMoveMulti = 0.3f;
    timeToSwipe = 0.1f;
    strikeI = (strikeI + 1) % strikeCount;
    timeInvincible = 0.15f;
}

void Actor::Roll()
{
    if (inventory.hearts == 0)
        return;
    model.PlayAnimation(anims[ActorAnim_Roll], 0.0, 30.0f);
    rollDownTime = 0.6f;
    animDownTime = 0.3f;
    animMoveMulti = 1.3f;
    timeInvincible = 0.5f;
}

void Actor::TakeSwipe()
{
    if (inventory.hearts == 0)
        return;
    FXer::Fire(this, FXer::GetSwordProjectile(inventory.holdItem));
}

void Actor::OnStrike(MapEntity *other, u8 power)
{
    if ((faction == ActorFaction_EVIL) != (other->GetFaction() == ActorFaction_EVIL))
    {
        if (MapEntityTypes::IsActor(other->type) && ((Actor*)other)->inventory.hearts <= power && inventory.holdItem == MapEntityType_SwordFire)
        {
            ((Actor*)other)->OnDead();
        }
        else
        {
            other->TakeHit(this, power);
        }
    }
    FXer::Fire(this, FXType_Dust, Vector3(0.0f, 1.5f, 0.0f));
}

void Actor::SetHoldItem(MapEntityType entityType)
{
    if (inventory.hearts == 0)
        return;
    MapEntityType existingType = holdItem != NULL ? holdItem->type : MapEntityType_None;
    if (existingType == entityType)
        return;

    if (holdItem != NULL)
    {
        delete holdItem;
        holdItem = NULL;
    }

    if (entityType != MapEntityType_None)
    {
        static MapEntityFactory factory;
        holdItem = factory.Create((MapEntityType)entityType);
        holdItem->Start();
    }

    inventory.holdItem = entityType;
}

void Actor::Face(Entity *other)
{
    if (other == NULL || inventory.hearts == 0)
        return;

    dirMove = other->transform.position - transform.position;
    dirMove.y = 0.0f;
    dirMove = Maths::Normalize(dirMove);
}

void Actor::SetBehaviour(ActorBehaviour *_behaviour, bool override)
{
    if (override || _behaviour == NULL)
    {
        for (size_t i = 0; i<behaviours.size(); i++)
        {
            delete behaviours[i];
        }
        behaviours.clear();
        behaviour = NULL;
    }
    if (_behaviour != NULL)
    {
        behaviours.push_back(_behaviour);
    }
}

void Actor::Chill()
{
    if (inventory.hearts == 0)
        return;
    AttackManager::Remove(this);
    if (behaviour != NULL)
    {
        behaviour->Chill();
    }
}

Actor * Actor::FindTarget() const
{
    if (scene != NULL)
    {
        u8 factionMask = GetOpposisingFactionMask();
        Actor * result = (Actor*)scene->GetNearestEntity(transform.position, factionMask);
        if (result != NULL && MapEntityTypes::IsActor(result->type))
            return result;
    }
    return NULL;
}

u8 Actor::GetOpposisingFactionMask() const
{
    if (faction == ActorFaction_GOOD || faction == ActorFaction_PLAYER)
        return ActorFaction_EVIL;
    return ActorFaction_GOOD | ActorFaction_PLAYER;
}

bool Actor::IsWithinRange(Vector3 targetPosition, float range, float minDot) const
{
    Vector3 diff = targetPosition - transform.position;
    diff.y = 0.0f;
    float dot = Maths::Dot(Maths::Normalize(diff), transform.Forwards());
    if (dot < minDot)
        return false;
    return Maths::IsWithinDistance(diff, range * dot);
}

bool Actor::IsWithinRange(Entity * target, float range, float minDot) const
{
    if (target == NULL)
        return false;
    return IsWithinRange(target->transform.position, range, minDot);
}

void Actor::OnCollision(Entity * other)
{
    MapEntity::OnCollision(other);
    if (behaviour != NULL)
    {
        behaviour->OnCollision(other);
    }
}

bool Actor::TakeHit(Entity *other, u8 power)
{
    if (knockTime > 0.0f)
        return false;

    if (timeInvincible > 0.0f)
        return false;

    if (inventory.hearts == 0)
        return false;

    attacker = (Actor*)other;

    Knock(other);

    Camera * cam = Camera::Current();
    if (cam != NULL)
    {
        cam->Shake(0.5f);
    }

    if (anims[ActorAnim_Hit] != NULL)
    {
        model.PlayAnimation(anims[ActorAnim_Hit]);
    }
    else
    {
        model.PlayAnimation(anims[ActorAnim_Idle]);
    }

    if (inventory.hearts > power)
    {
        inventory.hearts -= power;
        if (behaviour != NULL)
        {
            behaviour->TakeHit();
        }
    }
    else
    {
        Die();
    }

    animDownTime = MapEntity_KNOCK_TIME;
    timeInvincible = 0.5f;
    return true;
}

void Actor::Die()
{
    if (inventory.hearts == 0)
        return;
    SetBehaviour(NULL);
    if (anims[ActorAnim_Die] != NULL)
    {
        if (state == ENTITY_STATE_ADDED)
        {
            model.fps = 1.0f;
            SetAnimation(ActorAnim_Die, anims[ActorAnim_Die]->frameCount, INFINITY);
        }
        else
        {
            PlayAnimation(ActorAnim_Die, INFINITY);
        }
    }
    inventory.hearts = 0;
    timeToDie = 1.0f;
    if (collider != NULL)
    {
        collider->mode = CollisionMode_None;
    }
    if (permadeath)
    {
        map->FlagSet(info->id, true);
    }
    map->UpdateFlags();
}

void Actor::PlayAnimation(ActorAnim animation, float duration, float fps)
{
    if (inventory.hearts == 0)
        return;
    model.PlayAnimation(anims[animation], 0.0f, fps);
    animDownTime = duration;
    animPauseTime = 0.0f;
}

void Actor::SetAnimation(ActorAnim animation, float time, float pauseTime)
{
    if (inventory.hearts == 0)
        return;
    model.PlayAnimation(anims[animation], time);
    animPauseTime = pauseTime;
    animDownTime = pauseTime;
    model.Update(0.0f);
}

void Actor::OnDead()
{
    inventory.hearts = 0;
    AttackManager::Remove(this);
    if (removeOnDeath)
    {
        if (faction == ActorFaction_EVIL)
        {
            FXer::Fire(this, FXType_Evaporate);
            if (initHearts > 0)
            {
                for (u8 i=0; i<initHearts; i++)
                {
                    if (i == 0 && Actor::IsActor(attacker) && attacker->inventory.hearts < attacker->inventory.maxHearts && (Maths::RandI(2) == 0 || attacker->inventory.hearts <= 2))
                    {
                        FXer::Fire(this, FXType_Heart);
                    }
                    else
                    {
                        FXer::Fire(this, FXType_Chicken);
                    }
                }
            }
        }
        SetEnabled(false);
    }
    map->UpdateFlags();
}

void Actor::OutOfMap()
{
    if (canDespawn)
        OnDead();
}

Vector3 Actor::GetMoveVelocity(float dt)
{
    if (behaviour != NULL)
    {
        return behaviour->GetMoveVelocity(dt);
    }
    if (walkPoint != WalkPoint_NONE)
    {
        Vector3 diff = map->GetPosition(walkPoint.x, walkPoint.z);
        diff -= transform.position;
        diff.y = 0.0f;
        return Maths::Clamp(diff, 1.0f) * moveSpeed;
    }
    return MapEntity::GetMoveVelocity(dt);
}

void Actor::Render()
{
    Matrix mtx = transform.GetMatrix();
    model.Render(mtx);
    if (holdItem != NULL && boneHold != NULL)
    {
        mtx *= model.GetBonePose(boneHold, false);
        mtx = glm::rotate(mtx, PI * 0.5f, Vector3_RIGHT);
        holdItem->transform.SetMatrix(mtx);
        holdItem->Render();
    }
}