#include "IqmReader.h"
#include <desslog.h>
#include <string.h>
#include <math.h>
#include <Mesh.h>
#include <FileSystem.h>

IqmReader::IqmReader(){}

IqmReader::~IqmReader()
{
    CleanUp();
}

bool IqmReader::Load(const char * modelPath)
{
    SDL_RWops * file = FileSystem::FileOpen(modelPath, "r+b");
    if (file == NULL) 
    {
        Log_Error("Could not open IQM file %s\n", modelPath);
        return false;
    }

    uchar * buf = NULL;
    iqmheader hdr;
    if (FileSystem::FileRead(file, &hdr, sizeof(hdr), 1) == 0)
    {
        Log_Error("Could not read IQM header for %s\n", modelPath);
        return false;
    }

    if (HashEqual(hdr.magic, IQM_MAGIC) == false)
    {
        Log_Error("Invalid IQM header string %s\n", modelPath);
        return false;
    }

    lilswap(&hdr.version, (sizeof(hdr) - sizeof(hdr.magic))/sizeof(uint));
    if (hdr.version != IQM_VERSION)
    {
        Log_Error("IQM version %d not supported for %s\n", hdr.version, modelPath);
        return false;
    }

    if (hdr.filesize > (16<<20)) 
    {
        Log_Error("IQM file is too big (>16MB) for %s\n", modelPath);
        return false;
    }

    buf = new uchar[hdr.filesize];
    if (FileSystem::FileRead(file, buf + sizeof(hdr), 1, hdr.filesize - sizeof(hdr)) == 0)
    {
        Log_Error("Could not buffer data from IQM %s", modelPath);
        return false;
    }

    if (hdr.num_meshes > 0 && !LoadMeshData(modelPath, hdr, buf))
    {
        Log_Error("Could load IQM mesh for %s", modelPath);
        return false;
    }

    if (hdr.num_anims > 0 && !LoadAnimData(modelPath, hdr, buf))
    {
        Log_Error("Could load IQM animation for %s", modelPath);
        return false;
    }
 
    FileSystem::FileClose(file);
    return true;
}

void IqmReader::Clear()
{
    //CleanUp();
}

bool IqmReader::LoadAnimations(const char * animModelPath)
{
    SDL_RWops * file = FileSystem::FileOpen(animModelPath, "r+b");
    if (file == NULL) 
    {
        Log_Error("Could not open IQM animations file %s: %s\n", animModelPath, SDL_GetError());
        return false;
    }

    uchar* buf = NULL;
    iqmheader hdr;
    if (FileSystem::FileRead(file, &hdr, sizeof(hdr), 1) == 0)
    {
        Log_Error("Could not read IQM header for %s\n", animModelPath);
        return false;
    }

    if (HashEqual(hdr.magic, IQM_MAGIC) == false)
    {
        Log_Error("Invalid IQM header string %s\n", animModelPath);
        return false;
    }

    lilswap(&hdr.version, (sizeof(hdr) - sizeof(hdr.magic)) / sizeof(uint));
    if (hdr.version != IQM_VERSION)
    {
        Log_Error("IQM version %d not supported for %s\n", hdr.version, animModelPath);
        return false;
    }

    if (hdr.filesize > (16<<20)) 
    {
        Log_Error("IQM animations file is too big (>16MB) for %s", animModelPath);
        return false;
    }

    buf = new uchar[hdr.filesize];
    if (FileSystem::FileRead(file, buf + sizeof(hdr), 1, hdr.filesize - sizeof(hdr)) == 0)
    {
        Log_Error("Could not buffer data from IQM %s", animModelPath);
        return false;
    }

    if (hdr.num_anims > 0 && !LoadAnimData(animModelPath, hdr, buf))
    {
        Log_Error("Could load IQM animation for %s", animModelPath);
        return false;
    }
 
    SDL_RWclose(file);
    return true;
}

bool IqmReader::LoadMeshData(const char *filename, const iqmheader &hdr, uchar *buf)
{
    if (meshdata) return false;

    lilswap((uint *)&buf[hdr.ofs_vertexarrays], hdr.num_vertexarrays*sizeof(iqmvertexarray)/sizeof(uint));
    lilswap((uint *)&buf[hdr.ofs_triangles], hdr.num_triangles*sizeof(iqmtriangle)/sizeof(uint));
    lilswap((uint *)&buf[hdr.ofs_meshes], hdr.num_meshes*sizeof(iqmmesh)/sizeof(uint));
    lilswap((uint *)&buf[hdr.ofs_joints], hdr.num_joints*sizeof(iqmjoint)/sizeof(uint));
    if(hdr.ofs_adjacency) lilswap((uint *)&buf[hdr.ofs_adjacency], hdr.num_triangles*sizeof(iqmtriangle)/sizeof(uint));

    meshdata = buf;
    nummeshes = hdr.num_meshes;
    numtris = hdr.num_triangles;
    numverts = hdr.num_vertexes;
    numjoints = hdr.num_joints;
    outposition = new float[3*numverts];
    outnormal = new float[3*numverts];
    outtangent = new float[3*numverts];
    outbitangent = new float[3*numverts];
    outframe = new iqmgeom::Matrix3x4[hdr.num_joints];

    const char *str = hdr.ofs_text ? (char *)&buf[hdr.ofs_text] : "";
    iqmvertexarray *vas = (iqmvertexarray *)&buf[hdr.ofs_vertexarrays];
    for(int i = 0; i < (int)hdr.num_vertexarrays; i++)
    {
        iqmvertexarray &va = vas[i];
        switch(va.type)
        {
        case IQM_POSITION: if(va.format != IQM_FLOAT || va.size != 3) return false; inposition = (float *)&buf[va.offset]; lilswap(inposition, 3*hdr.num_vertexes); break;
        case IQM_NORMAL: if(va.format != IQM_FLOAT || va.size != 3) return false; innormal = (float *)&buf[va.offset]; lilswap(innormal, 3*hdr.num_vertexes); break;
        case IQM_TANGENT: if(va.format != IQM_FLOAT || va.size != 4) return false; intangent = (float *)&buf[va.offset]; lilswap(intangent, 4*hdr.num_vertexes); break;
        case IQM_TEXCOORD: if(va.format != IQM_FLOAT || va.size != 2) return false; intexcoord = (float *)&buf[va.offset]; lilswap(intexcoord, 2*hdr.num_vertexes); break;
        case IQM_BLENDINDEXES: if(va.format != IQM_UBYTE || va.size != 4) return false; inblendindex = (uchar *)&buf[va.offset]; break;
        case IQM_BLENDWEIGHTS: if(va.format != IQM_UBYTE || va.size != 4) return false; inblendweight = (uchar *)&buf[va.offset]; break;
        case IQM_COLOR: if(va.format != IQM_UBYTE || va.size != 4) return false; incolor = (uchar *)&buf[va.offset]; break;
        }
    }
    tris = (iqmtriangle *)&buf[hdr.ofs_triangles];
    meshes = (iqmmesh *)&buf[hdr.ofs_meshes];
    joints = (iqmjoint *)&buf[hdr.ofs_joints];
    if(hdr.ofs_adjacency) adjacency = (iqmtriangle *)&buf[hdr.ofs_adjacency];

    baseframe = new iqmgeom::Matrix3x4[hdr.num_joints];
    inversebaseframe = new iqmgeom::Matrix3x4[hdr.num_joints];
    for(int i = 0; i < (int)hdr.num_joints; i++)
    {
        iqmjoint &j = joints[i];
        baseframe[i] = iqmgeom::Matrix3x4(iqmgeom::Quat(j.rotate).normalize(), iqmgeom::Vec3(j.translate), iqmgeom::Vec3(j.scale));
        inversebaseframe[i].invert(baseframe[i]);
        if(j.parent >= 0) 
        {
            baseframe[i] = baseframe[j.parent] * baseframe[i];
            inversebaseframe[i] *= inversebaseframe[j.parent];
        }
    }

    for(int i = 0; i < (int)hdr.num_meshes; i++)
    {
        iqmmesh &m = meshes[i];
    }

    return true;
}

bool IqmReader::LoadAnimData(const char *filename, const iqmheader &hdr, uchar *buf)
{
    if((int)hdr.num_poses != numjoints) return false;

    if(animdata)
    {
        if(animdata != meshdata) delete[] animdata;
        delete[] frames;
        animdata = NULL;
        anims = NULL;
        frames = 0;
        numframes = 0;
        numanims = 0;
    }        

    lilswap((uint *)&buf[hdr.ofs_poses], hdr.num_poses*sizeof(iqmpose)/sizeof(uint));
    lilswap((uint *)&buf[hdr.ofs_anims], hdr.num_anims*sizeof(iqmanim)/sizeof(uint));
    lilswap((ushort *)&buf[hdr.ofs_frames], hdr.num_frames*hdr.num_framechannels);
    if(hdr.ofs_bounds) lilswap((uint *)&buf[hdr.ofs_bounds], hdr.num_frames*sizeof(iqmbounds)/sizeof(uint));

    animdata = buf;
    numanims = hdr.num_anims;
    numframes = hdr.num_frames;

    const char *str = hdr.ofs_text ? (char *)&buf[hdr.ofs_text] : "";
    anims = (iqmanim *)&buf[hdr.ofs_anims];
    poses = (iqmpose *)&buf[hdr.ofs_poses];
    frames = new iqmgeom::Matrix3x4[hdr.num_frames * hdr.num_poses];
    ushort *framedata = (ushort *)&buf[hdr.ofs_frames];
    if(hdr.ofs_bounds) bounds = (iqmbounds *)&buf[hdr.ofs_bounds];

    for(int i = 0; i < (int)hdr.num_frames; i++)
    {
        for(int j = 0; j < (int)hdr.num_poses; j++)
        {
            iqmpose &p = poses[j];
            iqmgeom::Quat rotate;
            iqmgeom::Vec3 translate, scale;
            translate.x = p.channeloffset[0]; if(p.mask&0x01) translate.x += *framedata++ * p.channelscale[0];
            translate.y = p.channeloffset[1]; if(p.mask&0x02) translate.y += *framedata++ * p.channelscale[1];
            translate.z = p.channeloffset[2]; if(p.mask&0x04) translate.z += *framedata++ * p.channelscale[2];
            rotate.x = p.channeloffset[3]; if(p.mask&0x08) rotate.x += *framedata++ * p.channelscale[3];
            rotate.y = p.channeloffset[4]; if(p.mask&0x10) rotate.y += *framedata++ * p.channelscale[4];
            rotate.z = p.channeloffset[5]; if(p.mask&0x20) rotate.z += *framedata++ * p.channelscale[5];
            rotate.w = p.channeloffset[6]; if(p.mask&0x40) rotate.w += *framedata++ * p.channelscale[6];
            scale.x = p.channeloffset[7]; if(p.mask&0x80) scale.x += *framedata++ * p.channelscale[7];
            scale.y = p.channeloffset[8]; if(p.mask&0x100) scale.y += *framedata++ * p.channelscale[8];
            scale.z = p.channeloffset[9]; if(p.mask&0x200) scale.z += *framedata++ * p.channelscale[9];
            // Concatenate each pose with the inverse base pose to avoid doing this at animation time.
            // If the joint has a parent, then it needs to be pre-concatenated with its parent's base pose.
            // Thus it all negates at animation time like so: 
            //   (parentPose * parentInverseBasePose) * (parentBasePose * childPose * childInverseBasePose) =>
            //   parentPose * (parentInverseBasePose * parentBasePose) * childPose * childInverseBasePose =>
            //   parentPose * childPose * childInverseBasePose
            iqmgeom::Matrix3x4 m(rotate.normalize(), translate, scale);
            if(p.parent >= 0) frames[i*hdr.num_poses + j] = baseframe[p.parent] * m * inversebaseframe[j];
            else frames[i*hdr.num_poses + j] = m * inversebaseframe[j];
        }
    }
 
    animations.resize(hdr.num_anims);
    for(u16 i = 0; i < (u16)hdr.num_anims; i++)
    {
        iqmanim &a = anims[i];
        animations[i].frameFirst = a.first_frame;
        animations[i].frameCount = a.num_frames;
        strncpy(animations[i].name, &str[a.name], MODEL_NAME_LENGTH);
    }

    iqmjoint * animJoints = (iqmjoint *)&buf[hdr.ofs_joints];
    bones.resize(hdr.num_joints);
    for (u16 i = 0; i < (u16)hdr.num_joints; i++)
    {
        iqmjoint &j = animJoints[i];
        strncpy(bones[i].name, &str[j.name], MODEL_NAME_LENGTH);
    }
    
    return true;
}

void BlendF(float * dest, float * src, float blend)
{
    *dest = (1.0f - blend) * (*dest) + blend * (*src);
}

void BlendMat(iqmgeom::Matrix3x4 * dest, iqmgeom::Matrix3x4 * src, float blend)
{
    BlendF(&dest->a.w, &src->a.w, blend);
    BlendF(&dest->a.x, &src->a.x, blend);
    BlendF(&dest->a.y, &src->a.y, blend);
    BlendF(&dest->a.z, &src->a.z, blend);
    BlendF(&dest->b.w, &src->b.w, blend);
    BlendF(&dest->b.x, &src->b.x, blend);
    BlendF(&dest->b.y, &src->b.y, blend);
    BlendF(&dest->b.z, &src->b.z, blend);
    BlendF(&dest->c.w, &src->c.w, blend);
    BlendF(&dest->c.x, &src->c.x, blend);
    BlendF(&dest->c.y, &src->c.y, blend);
    BlendF(&dest->c.z, &src->c.z, blend);
}

void IqmReader::Animate(float time, ModelAnimData * animA, ModelAnimData * animB, float blend) const
{
    if (!numframes) return;

    u16 frameA1 = (int)floor(time);
    u16 frameA2 = frameA1 + 1;
    u16 frameB1 = frameA1;
    u16 frameB2 = frameA2;

    u16 frameAFirst = 0;
    u16 frameACount = numframes;
    if (animA != NULL)
    {
        frameAFirst = animA->frameFirst;
        frameACount = animA->frameCount;
        if (animA->loop == false && time >= frameACount - 1)
        {
            frameA1 = frameA2 = frameACount - 1;
        }
    }
    
    u16 frameBFirst = 0;
    u16 frameBCount = numframes;
    if (animB != NULL)
    {
        frameBFirst = animB->frameFirst;
        frameBCount = animB->frameCount;
        if (animB->loop == false && time >= frameBCount - 1)
        {
            frameB1 = frameB2 = frameBCount - 1;
        }
    }

    float frameAOffset = time - frameA1;
    float frameBOffset = time - frameB1;
    frameA1 = (frameA1 % frameACount) + frameAFirst;
    frameA2 = (frameA2 % frameACount) + frameAFirst;
    frameB1 = (frameB1 % frameBCount) + frameBFirst;
    frameB2 = (frameB2 % frameBCount) + frameBFirst;
    iqmgeom::Matrix3x4 *matA1 = &frames[frameA1 * numjoints],
                       *matA2 = &frames[frameA2 * numjoints];
    iqmgeom::Matrix3x4 *matB1 = &frames[frameB1 * numjoints],
                       *matB2 = &frames[frameB2 * numjoints];
    // Interpolate matrixes between the two closest frames and concatenate with parent matrix if necessary.
    // Concatenate the result with the inverse of the base pose.
    // You would normally do animation blending and inter-frame blending here in a 3D engine.
    for(int i = 0; i < numjoints; i++)
    {
        iqmgeom::Matrix3x4 mat = matA1[i]*(1 - frameAOffset) + matA2[i]*frameAOffset;
        if (animB != NULL && blend > 0.0f)
        {
            iqmgeom::Matrix3x4 matB = matB1[i]*(1 - frameBOffset) + matB2[i]*frameBOffset;
            BlendMat(&mat, &matB, blend);
        }
        if(joints[i].parent >= 0) outframe[i] = outframe[joints[i].parent] * mat;
        else outframe[i] = mat;
        WriteBone(&outframe[i], &joints[i], (ModelBoneData*)&bones[i]);
    }
    // The actual vertex generation based on the matrixes follows...
    const iqmgeom::Vec3 *srcpos = (const iqmgeom::Vec3 *)inposition, *srcnorm = (const iqmgeom::Vec3 *)innormal;
    const iqmgeom::Vec4 *srctan = (const iqmgeom::Vec4 *)intangent; 
    iqmgeom::Vec3 *dstpos = (iqmgeom::Vec3 *)outposition, *dstnorm = (iqmgeom::Vec3 *)outnormal, *dsttan = (iqmgeom::Vec3 *)outtangent, *dstbitan = (iqmgeom::Vec3 *)outbitangent; 
    const uchar *index = inblendindex, *weight = inblendweight;
    for(int i = 0; i < numverts; i++)
    {
        // Blend matrixes for this vertex according to its blend weights. 
        // the first index/weight is always present, and the weights are
        // guaranteed to add up to 255. So if only the first weight is
        // presented, you could optimize this case by skipping any weight
        // multiplies and intermediate storage of a blended matrix. 
        // There are only at most 4 weights per vertex, and they are in 
        // sorted order from highest weight to lowest weight. Weights with 
        // 0 values, which are always at the end, are unused.
        iqmgeom::Matrix3x4 mat = outframe[index[0]] * (weight[0]/255.0f);
        for(int j = 1; j < 4 && weight[j]; j++)
            mat += outframe[index[j]] * (weight[j]/255.0f);

        // Transform attributes by the blended matrix.
        // Position uses the full 3x4 transformation matrix.
        // Normals and tangents only use the 3x3 rotation part 
        // of the transformation matrix.
        *dstpos = mat.transform(*srcpos);

        // Note that if the matrix includes non-uniform scaling, normal vectors
        // must be transformed by the inverse-transpose of the matrix to have the
        // correct relative scale. Note that invert(mat) = adjoint(mat)/determinant(mat),
        // and since the absolute scale is not important for a vector that will later
        // be renormalized, the adjoint-transpose matrix will work fine, which can be
        // cheaply generated by 3 cross-products.
        //
        // If you don't need to use joint scaling in your models, you can simply use the
        // upper 3x3 part of the position matrix instead of the adjoint-transpose shown 
        // here.
        iqmgeom::Matrix3x3 matnorm(mat.b.cross3(mat.c), mat.c.cross3(mat.a), mat.a.cross3(mat.b));

        *dstnorm = matnorm.transform(*srcnorm);
        // Note that input tangent data has 4 coordinates, 
        // so only transform the first 3 as the tangent vector.
        *dsttan = matnorm.transform(iqmgeom::Vec3(*srctan));
        // Note that bitangent = cross(normal, tangent) * sign, 
        // where the sign is stored in the 4th coordinate of the input tangent data.
        *dstbitan = dstnorm->cross(*dsttan) * srctan->w;

        srcpos++;
        srcnorm++;
        srctan++;
        dstpos++;
        dstnorm++;
        dsttan++;
        dstbitan++;

        index += 4;
        weight += 4;
    }
}

void IqmReader::WriteVerts(float * in_vertex, float * in_normal, Vector3 * out_vertex, Vector3 * out_normal) const
{
    out_vertex->x = in_vertex[0] * scale;
    out_vertex->y = in_vertex[2] * scale;
    out_vertex->z = in_vertex[1] * -scale;
    out_normal->x = in_normal[0];
    out_normal->y = in_normal[2];
    out_normal->z = -in_normal[1];
}

void IqmReader::Process(Mesh * mesh, float * time, ModelAnimData * animA, ModelAnimData * animB, float blend) const
{ 
    if (mesh->GetVertexCount() != numverts)
    {
        mesh->SetVertexCount(numverts);
        mesh->SetIndexCount(numtris * 3);
        for (u16 i=0; i<numtris; i++)
        {
            mesh->indices[i * 3] = tris[i].vertex[0];
            mesh->indices[i * 3 + 1] = tris[i].vertex[1];
            mesh->indices[i * 3 + 2] = tris[i].vertex[2];
        }
        for (u16 i=0; i<numverts; i++)
        {
            WriteVerts(&inposition[i * 3], &innormal[i * 3], &mesh->vertices[i], &mesh->normals[i]);
            mesh->uvs[i].x = intexcoord[i * 2];
            mesh->uvs[i].y = intexcoord[i * 2 + 1];
        }
    }

    if (numframes < 1)
        return;

    Animate(*time * 30.0f, animA, animB, blend);

    for(u16 i = 0; i < numverts; i++)
    {
        WriteVerts(&outposition[i * 3], &outnormal[i * 3], &mesh->vertices[i], &mesh->normals[i]);
    }
}

inline void IqmReader::WriteVec3(iqmgeom::Vec3 * in, Vector3 * out) const
{
    out->x = (*in)[0] * scale;
    out->y = (*in)[2] * scale;
    out->z = (*in)[1] * -scale;
}

void IqmReader::WriteBone(iqmgeom::Matrix3x4 * mtx, iqmjoint * joint, ModelBoneData * out) const
{
    iqmgeom::Vec3 v(baseframe[joint->parent].transform(iqmgeom::Vec3(joint->translate)));
    v = mtx->transform(v);
    WriteVec3(&v, &out->transform.position);
    iqmgeom::Quat q(*mtx);
    out->transform.rotation.w = q.w;
    out->transform.rotation.x = q.x;
    out->transform.rotation.y = q.z;
    out->transform.rotation.z = -q.y;
}

void IqmReader::CleanUp()
{
    delete[] outposition;
    delete[] outnormal;
    delete[] outtangent;
    delete[] outbitangent;
    delete[] baseframe;
    delete[] inversebaseframe;
    delete[] outframe;
    delete[] frames;
}