#include <Transform.h>
#include <glm/gtx/quaternion.hpp>
#include <glm/ext/quaternion_transform.hpp>
#include <glm/gtx/matrix_decompose.hpp>

Transform::Transform() : 
    position(Vector3(0.0f)),
    rotation(Vector3(0.0f)),
    scale(Vector3(1.0f))
 {}

Transform & Transform::Rotate(Vector3 axis, float angle)
{
    rotation = glm::rotate(rotation, angle, axis);
    return *this;
}

Transform & Transform::Rotate(Quat rotation)
{
    rotation *= rotation;
    return *this;
}

Transform & Transform::LookAt(Vector3 target, Vector3 up)
{
    return LookDirection(target - position);
}

Transform & Transform::LookDirection(Vector3 direction, Vector3 up)
{
    rotation = glm::quatLookAt(glm::normalize(direction), up);
    return *this;
}

Transform & Transform::Translate(Vector3 offset)
{
    position += offset;
    return *this;
}

Transform & Transform::SetPosition(Vector3 value)
{
    position = Vector3(value);
    return *this;
}

Transform & Transform::SetRotation(Quat value)
{
    rotation = Quat(value);
    return *this;
}

Transform & Transform::SetScale(Vector3 value)
{
    scale = Vector3(value);
    return *this;
}

Transform & Transform::SetScale(float value)
{
    scale = Vector3(value);
    return *this;
}

void Transform::SetMatrix(Matrix mtx)
{
    glm::vec3 a;
    glm::vec4 b;
    glm::decompose((const glm::mat4x4)mtx, scale, rotation, position, a, b);
}

Matrix Transform::GetMatrix() const
{
    Matrix mtx = glm::translate(Matrix(1.0f), position);
    mtx *= glm::mat4_cast(rotation);
    mtx = glm::scale(mtx, scale);
    return mtx;
}

Vector3 Transform::Forwards() const
{
    return glm::rotate(rotation, Vector3_BACK);
}

Vector3 Transform::Up() const
{
    return glm::rotate(rotation, Vector3_UP);
}

Vector3 Transform::Right() const
{
    return glm::rotate(rotation, Vector3_RIGHT);
}