#include <Textbox.h>
#include <string.h>
#include <Font.h>
#include <Material.h>

#define GLYPH_WIDTH 8.0f
#define GLYPH_HEIGHT 15.0f
#define TEXTURE_SIZE 128.0f 

Textbox::Textbox() :
    isDirty(true),
    fontSize(16),
    color(1.0f, 1.0f, 1.0f)
{}

Textbox::~Textbox()
{
    Material::Free(material);
}

void Textbox::Added()
{
    if (font == NULL)
    {
        SetFont("fonts/pixel");
    }
    renderer.SetRenderSpace(RenderSpace_Screen);
    isDirty = true;
}

void Textbox::SetFont(const char * pathPrefix)
{
    Material::Free(material);
    font = Font::Load(pathPrefix);
    if (font.IsValid())
    {
        mesh.SetMaterial(Material::Get(font->GetMaterial()));
        isDirty = true;
    }
    else
    {
        Log_Error("Couldn't set font on textbox: %s\n", pathPrefix);
    }
}

void Textbox::SetText(const char * newText)
{
    if (newText == NULL || newText[0] == '\0')
    {
        if (text[0] != '\0')
        {
            text[0] = '\0';
            isDirty = true;
        }
    }
    else
    {
        if (strncmp(newText, text, Textbox_MAX_TEXT_LENGTH) != 0)
        {
            strncpy(text, newText, Textbox_MAX_TEXT_LENGTH);
            text[Textbox_MAX_TEXT_LENGTH - 1] = '\0';
            isDirty = true;
        }
    }
}

void Textbox::SetAlignment(Vector2 value)
{
    if (alignment != value)
    {
        alignment = value;
        isDirty = true;
    }
}

void Textbox::SetFontSize(u8 value)
{
    if (fontSize != value)
    {
        fontSize = value;
        isDirty = true;
    }
}

void Textbox::SetMaxWidth(float value)
{
    if (maxWidth != value)
    {
        maxWidth = value;
        isDirty = true;
    }
}

void Textbox::SetMaxDisplayLength(u16 value)
{
    if (value > textLength)
    {
        value = textLength;
    }
    if (maxDisplayLength != value)
    {
        maxDisplayLength = value;
        isDirty = true;
    }
}

void Textbox::SetColor(Vector3 value)
{
    if (color != value)
    {
        color = value;
        isDirty = true;
    }
}

u16 Textbox::GetTextLength() 
{
    if (isDirty)
    {
        Redraw();
    }
    return textLength;
}

void Textbox::Render()
{
    if (isDirty)
    {
        Redraw();
    }

    if (font == NULL)
        return;

    if (text == NULL || text[0] == '\0')
        return;

    if (mesh.GetVertexCount() < 4)
        return;
        
    renderer.Render(&mesh, transform.GetMatrix());
}

#define SPACE_MULTI 1.0f
#define TAB_MULTI 4.0f
#define MAX_LINES 8

void Textbox::Redraw()
{
    isDirty = false;
    mesh.ClearGeometry();

    if (font == NULL)
        return;

    if (text == NULL || text[0] == '\0')
        return;
    
    float scaleU = 1.0f / (font->props.textureWidth > 0 ? (float)font->props.textureWidth : 1.0f);
    float scaleV = 1.0f / (font->props.textureHeight > 0 ? (float)font->props.textureHeight : 1.0f);
    float rescale = (float)fontSize / (float)(font->props.size > 0 ? font->props.size : 0);
    float letterSpacing = floor(fontSize * -0.1f);
    float lineHeight = fontSize * 1.5f;

    // find line dimensions
    float lineWidths[MAX_LINES];
    int lineBreaks[MAX_LINES];
    u16 lineCount = 0;
    u16 lastSpaceI = 0;
    float lastSpaceWidth = 0.0f;
    char c;
    width = 0.0f;
    height = fontSize;
    textLength = 0;
    const FontChar * fontChar = NULL;
    for (u16 i = 0; i < Textbox_MAX_TEXT_LENGTH; i++)
    {
        c = text[i];
        if (c == '\0')
            break;

        if (c == ' ')
        {
            lastSpaceWidth = width;
            lastSpaceI = i;
            width += fontSize * SPACE_MULTI;
            textLength++;
        }
        else if (c == '\t')
        {
            lastSpaceWidth = width;
            lastSpaceI = i;
            width += fontSize * TAB_MULTI;
            textLength++;
        } 
        else if (c == '\n')
        {
            lineBreaks[lineCount] = i;
            lineWidths[lineCount] = width;
            width = 0.0f;
            lastSpaceWidth = 0.0f;
            height += lineHeight;
            lineCount++;
            textLength++;
        }
        else
        {
            fontChar = font->GetChar(c);
            textLength++;
            width += (fontChar->w) * rescale + letterSpacing;
            if (maxWidth > 0.0f && width > maxWidth && lineCount < MAX_LINES - 1)
            {
                width -= lastSpaceWidth;
                lineWidths[lineCount] = lastSpaceWidth;
                lineBreaks[lineCount] = lastSpaceI;
                height += lineHeight;
                lineCount++;
                lastSpaceWidth = 0.0f;
            }
        }
    }
    lineWidths[lineCount] = width;
    lineBreaks[lineCount] = Textbox_MAX_TEXT_LENGTH;
    lineCount++;

    // get bounding size
    if (lineCount > 1)
    {
        width = 0.0f;
        for (u16 i = 0; i<lineCount; i++)
        {
            if (lineWidths[i] > width)
            {
                width = lineWidths[i];
            }
        }
    }

    // draw mesh
    
    std::vector<Vector3> vertices;
    vertices.resize(textLength * 4);
    std::vector<Vector2> uvs;
    uvs.resize(textLength * 4);
    std::vector<Vector3> colors;
    colors.resize(textLength * 4);
    std::vector<u32> indices;
    indices.resize(textLength * 6);

    u16 lineI = 0;
    float offsetX = alignment.x * - lineWidths[0];
    float x = 0.0f;
    float y = alignment.y * -height;
    float top, right, bottom, left;
    for (u16 i = 0; i < textLength; i++)
    {
        if (i == lineBreaks[lineI])
        {
            x = 0.0f;
            y += lineHeight;
            lineI++;
            offsetX = alignment.x * - lineWidths[lineI];
        }

        c = text[i];
        if (c == ' ')
        {
            x += fontSize * SPACE_MULTI;
            continue;
        }

        if (c == '\t')
        {
            x += fontSize * TAB_MULTI;
            continue;
        }

        fontChar = font->GetChar(c);
    
        left = x + offsetX + fontChar->offX * rescale;
        top = y + fontChar->offY * rescale;
        right = left + fontChar->w * rescale;
        bottom = top + fontChar->h * rescale;
        vertices[i * 4].x = left;
        vertices[i * 4].y = top;
        vertices[i * 4 + 1].x = right;
        vertices[i * 4 + 1].y = top;
        vertices[i * 4 + 2].x = right;
        vertices[i * 4 + 2].y = bottom;
        vertices[i * 4 + 3].x = left;
        vertices[i * 4 + 3].y = bottom;

        left = fontChar->x * scaleU;
        top = fontChar->y * scaleV;
        right = left + fontChar->w * scaleU;
        bottom = top + fontChar->h * scaleV;

        uvs[i * 4].x = left;
        uvs[i * 4].y = top;;
        uvs[i * 4 + 1].x = right;
        uvs[i * 4 + 1].y = top;
        uvs[i * 4 + 2].x = right;
        uvs[i * 4 + 2].y = bottom;
        uvs[i * 4 + 3].x = left;
        uvs[i * 4 + 3].y = bottom;

        colors[i * 4] = color;
        colors[i * 4 + 1] = color;
        colors[i * 4 + 2] = color;
        colors[i * 4 + 3] = color;

        indices[i * 6] = i * 4;
        indices[i * 6 + 1] = i * 4 + 1;
        indices[i * 6 + 2] = i * 4 + 3;
        indices[i * 6 + 3] = i * 4 + 3;
        indices[i * 6 + 4] = i * 4 + 1;
        indices[i * 6 + 5] = i * 4 + 2;

        x += (fontChar->w) * rescale + letterSpacing;

        if (maxDisplayLength > 0 && i > maxDisplayLength)
            break;
    }

    mesh.AddVertices(vertices);
    mesh.AddUvs(uvs);
    mesh.AddColors(colors);
    mesh.AddIndices(indices);
}