#include "SoundPlayer.h"
#include <SDL2/SDL_mixer.h>
#include <FileSystem.h>
#include <desslog.h>

SoundPlayer & SoundPlayer::Inst()
{
    static SoundPlayer inst;
    return inst;
}

SoundPlayer::SoundPlayer()
{
    if (Init() == false)
    {
        isMuted = true;
    }
}

bool SoundPlayer::Init()
{
    if (SDL_Init(SDL_INIT_AUDIO) < 0)
    {
        Log_Critical("Couldn't init audio: %s\n", SDL_GetError());
        return false;
    }

    if (Mix_OpenAudio(44100, AUDIO_S16SYS, 2, 512) < 0)
    {
        Log_Critical("Unable to open audio: %s\n", SDL_GetError());
        return false;
    }

    if (Mix_AllocateChannels(SOUND_CHANNEL_COUNT) < 0)
    {
        Log_Critical("Unable to allocate mixing channels: %s\n", SDL_GetError());
        return false;
    }

    return true;
}

void SoundPlayer::Play(const char * path, float volume, u16 flags)
{
    if (isMuted)
        return;
    SoundFile * file = GetFile(path);
    if (file == NULL)
    {
        Log_Error("Can't load file %s\n", path);
        return;
    }
    u32 tick = SDL_GetTicks();
    if ((tick - file->lastPlayTick) < 60)
    {
        return;
    }
    file->lastPlayTick = tick;
    u8 channel = nextChannel;
    nextChannel = (nextChannel + 1) % SOUND_CHANNEL_COUNT;
    Mix_HaltChannel(channel);
    u8 appliedVolume = volume > 0.0f ? volume <= 1.0f ? (u8)(MIX_MAX_VOLUME * volume) : MIX_MAX_VOLUME : 0;
    Mix_Volume(channel, appliedVolume);
    Mix_PlayChannel(channel, file->data, 0);
}

SoundFile * SoundPlayer::GetFile(const char * path)
{
    HashInt hash = Hash(path);
    SoundFile * file = NULL;
    for (std::vector<SoundFile>::iterator it = files.begin(); it != files.end(); ++it)
    {
        if (it->hash == hash)
        {
            file = &(*it);
            break;
        }
    }
    if (file == NULL)
    {
        std::string fullPath = FileSystem::GetAbsolutePath(path);
        Mix_Chunk * chunk = Mix_LoadWAV(fullPath.c_str());
        if (chunk == NULL)
            return NULL;

        files.push_back(SoundFile());
        file = &files.back();
        file->hash = hash;
        file->data = chunk;
    }
    return file;
}

void SoundPlayer::SetMuted(bool value)
{
    isMuted = value;
    if (!value)
    {
        Stop();
    }
}

void SoundPlayer::Stop()
{
    for (u8 i=0; i<SOUND_CHANNEL_COUNT; i++)
    {
        Mix_HaltChannel(i);
    }
}
