#include <Scene.h>
#include <Entity.h>
#include <Maths.h>

Scene::Scene()
{

}

Scene::~Scene()
{
    if (state != SceneState_Unloaded)
    {
        OnUnload();
        state = SceneState_Unloaded;
    }
    for (std::vector<Entity*>::iterator it = entities.begin(); it != entities.end(); ++it) 
    {
        if ((*it) != NULL)
        {
            (*it)->scene = NULL;
        }
    }
    entities.clear();
}

void Scene::AddEntity(Entity* entity)
{
    if (entity == NULL)
        return;
    if (FindEntity(entity) != entities.end())
        return;
    if (entity->scene != NULL)
    {
        entity->scene->RemoveEntity(entity);
    }
    entity->scene = this;
    entity->entityOwner = this;
    entities.push_back(entity);
    entity->state = ENTITY_STATE_ADDED;
    entity->Added();
}

Entity ENTITY_REMOVE_DUMMY;

void Scene::RemoveEntity(Entity* entity)
{
    if (entity == NULL || entity == &ENTITY_REMOVE_DUMMY)
        return;

    entity->state = ENTITY_STATE_REMOVED;
    entity->Removed();
    entity->scene = NULL;

    for (u16 i=0; i<entities.size(); i++)
    {
        if (entities[i] == entity)
        {
            entities[i] = &ENTITY_REMOVE_DUMMY;
        }
    }
}

std::vector<Entity*>::iterator Scene::FindEntity(Entity* entity)
{
    if (state != SceneState_Loaded)
        return entities.end();
        
    for (std::vector<Entity*>::iterator it = entities.begin(); it != entities.end(); ++it) 
    {
        if (*it == entity)
            return it;
    }
    return entities.end();
}

void Scene::Update(float dt)
{
    if (state != SceneState_Loaded)
        return;

    isUpdating = true;
    Entity * entity;
    size_t entityCount = entities.size();
    for (size_t i = 0; i < entityCount; i++)
    {
        entity = entities[i];
        if (entity == NULL)
            continue;

        if (entity->state == ENTITY_STATE_ADDED)
        {
            entity->Start();
            entity->state = ENTITY_STATE_ENABLED;
            entity->OnEnabled();
        }
        if (entity->state == ENTITY_STATE_ENABLED)
        {
            entity->Update(dt);
        }
        else if (entity->state == ENTITY_STATE_REMOVED)
        {
            entity->OnDisabled();
        }
        if (state == SceneState_Unloaded)
            return;
    }
    entityCount = entities.size();
    if (entityCount > 0)
    {
        for (size_t i = entityCount; i--; )
        {
            entity = entities[i];
            if (entity == NULL || entity->state >= ENTITY_STATE_REMOVED || entity == &ENTITY_REMOVE_DUMMY)
            {
                if (entity != NULL && entity != &ENTITY_REMOVE_DUMMY && entity->state == ENTITY_STATE_DESTROYING)
                {
                    delete entity;
                }
                entities.erase(entities.begin() + i);
            }
        }
    }
    isUpdating = false;
}

Entity * Scene::GetEntity(u16 index) const
{
    if (index >= entities.size())
        return NULL;
    Entity * result = entities[index];
    if (result == NULL || result->state >= ENTITY_STATE_REMOVED)
        return NULL;
    return result;
}

bool Scene::HasEntity(const Entity * entity) const
{
    for (std::vector<Entity*>::const_iterator it = entities.begin(); it != entities.end(); ++it) 
    {
        if (*it == entity)
            return true;
    }
    return false;
}

Entity * Scene::GetNearestEntity(Vector3 position, u8 factionMask, float distance)
{
    if (state != SceneState_Loaded)
        return NULL;
        
    float nearestSq = distance * distance;
    Entity * nearest = NULL;
    for (std::vector<Entity*>::iterator it = entities.begin(); it != entities.end(); ++it) 
    {
        if (*it == NULL || (*it)->state != ENTITY_STATE_ENABLED)
            continue;

        if ((*it)->transform.position == position)
            continue;

        if (factionMask > 0 && (factionMask & (*it)->faction) == 0)
            continue;

        float nearSq = Maths::LengthSq(position - (*it)->transform.position); 
        if (nearSq > nearestSq)
            continue;

        nearestSq = nearSq;
        nearest = *it;
    }
    return nearest;
}

void Scene::Render()
{
    if (state != SceneState_Loaded)
        return;
        
    isUpdating = true;
    i16 queue = INT16_MIN, next = INT16_MIN;
    Entity * entity = NULL;
    while (true)
    {
        for (u16 i = 0, lim = GetEntityCount(); i < lim; i++)
        {
            entity = entities[i];
            if (entity == NULL)
                continue;

            if (entity->isVisible == false)
                continue;

            if (entity->renderQueue == queue)
            {
                if (entity->state == ENTITY_STATE_ENABLED)
                {
                    entity->Render();
                }
            }
            else if (entity->renderQueue > queue)
            {
                if (next == INT16_MIN || entity->renderQueue < next)
                {
                    next = entity->renderQueue;
                }
            }
        }
        if (next == INT16_MIN)
            break;
        queue = next;
        next = INT16_MIN;
    }
    isUpdating = false;
}