
#ifndef _DESS_RENDERWINDOW_H
#define _DESS_RENDERWINDOW_H

#include <desslibs.h>
#include <vector>
#include <Camera.h>
#include <Mesh.h>

#define VIEWPORT_DEFAULT_WIDTH 600
#define VIEWPORT_DEFAULT_HEIGHT 360

class Material;
class RenderTexture;
class Shader;

enum RenderMode
{
    RenderMode_Standard = 0,
    RenderMode_Wireframe = 1,
    RenderMode_Depthless = 2
};

enum DisplayMode
{
    DisplayMode_Fixed,
    DisplayMode_Retro,
    DisplayMode_Modern
};

class RenderWindow
{
private:
    static RenderWindow * current;

    u32 renderWidth = 0;
    u32 renderHeight = 0;
    u32 windowWidth = 0;
    u32 windowHeight = 0;
    SDL_Window * window = NULL;
    SDL_GLContext context;
    bool isFullScreen = false;
    GLuint vertexArrayId = 0;
    const Shader * defaultShader = NULL;
    Material * defaultMaterial = NULL;
    RenderMode renderMode = RenderMode_Standard;
    DisplayMode displayMode = DisplayMode_Fixed;
    bool skipNext = false;
    u32 blankUntil = 0;

    GLuint postFramebuffer = 0;
    GLuint postTexture = 0;
    GLuint postDepthBuffer = 0;
    GLenum postDrawBuffers[1];
    GLuint postQuadBufferVertex = 0;
    GLuint postQuadBufferUV = 0;
    GLuint postShaderId = 0;
    GLuint postTextureId = 0;
    
    GLuint currentPostShaderId = 0;

public:
    static RenderWindow * Current();

    void Init(const char * title);
    void InitWindow(const char * title);
    void InitPostProcessing();

    void Clear();
    void Prepare();
    void Present();
    void SkipNext();
    void Resize(u32 width, u32 height);
    void Blank(u32 milliseconds);

    void SetFullScreen(bool nowFullScreen);

    void SetRenderMode(RenderMode renderMode);

    void SetPostShader(const char * shaderPath);

    void RenderData(const GLfloat * vertices, u32 vertexCount, const GLfloat * uvs = NULL, u32 uvCount = 0, const GLfloat * normals = NULL, u32 normalCount = 0, const u32 * indices = NULL, u32 indexCount = 0, const Material * material = NULL, const Matrix * matrix = NULL);
    
    void RenderMesh(const Mesh * mesh, const Matrix * matrix);

    inline u32 GetViewportWidth() const { return VIEWPORT_DEFAULT_WIDTH; }
    inline u32 GetViewportHeight() const { return VIEWPORT_DEFAULT_HEIGHT; }
    inline Vector3 GetViewportPos(float x, float y) const { 
        return Vector3(
            (float)VIEWPORT_DEFAULT_WIDTH * x,
            (float)VIEWPORT_DEFAULT_HEIGHT * y,
            0.0f
        );
    }
    inline bool IsFullScreen() const { return isFullScreen; }
    inline SDL_Window * GetWindow() const { return window; }
    inline Matrix GetViewMatrix() const { return mtxView; }
    inline Matrix GetProjectionMatrix() const { return mtxProjection; }
    inline RenderMode GetRenderMode() const { return renderMode; }
    inline const Shader * GetDefaultShader() const { return defaultShader; }
    inline const Material * GetDefaultMaterial() const { return defaultMaterial; }
    inline DisplayMode GetDisplayMode() const { return displayMode; }
    void SetDisplayMode(DisplayMode value) { displayMode = value; }

private:
    void UpdateProjection(const Matrix * matrix);

    Matrix mtxView;
    Matrix mtxProjection;
};

#endif