#include <RenderWindow.h>
#include <FileSystem.h>
#include <string>
#include <glm/gtc/matrix_transform.hpp>
#include <Transform.h>
#include <Shader.h>

RenderWindow * RenderWindow::current = NULL;

u32 default_flags = SDL_WINDOW_OPENGL | SDL_WINDOW_RESIZABLE;

void RenderWindow::Init(const char * title)
{
    InitWindow(title);
    
	glGenVertexArrays(1, &vertexArrayId);
	glBindVertexArray(vertexArrayId);

	defaultShader = Shader::LoadShader("shaders/missing.dsh");
    defaultMaterial = new Material();
    defaultMaterial->SetShader(defaultShader);

    // Enable depth test
    glEnable(GL_DEPTH_TEST);
    // Accept fragment if it closer to the camera than the former one
    glDepthFunc(GL_LESS);
    
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

    InitPostProcessing();
}

void RenderWindow::InitPostProcessing()
{

    // The framebuffer, which regroups 0, 1, or more textures, and 0 or 1 depth buffer.
    if (postFramebuffer)
    {
        glDeleteFramebuffers(1, &postFramebuffer);
    }
	postFramebuffer = 0;
	glGenFramebuffers(1, &postFramebuffer);
	glBindFramebuffer(GL_FRAMEBUFFER, postFramebuffer);

	// The texture we're going to render to
    if (postTexture)
    {
        glDeleteTextures(1, &postTexture);
    }
	glGenTextures(1, &postTexture);
	glBindTexture(GL_TEXTURE_2D, postTexture);
	glTexImage2D(GL_TEXTURE_2D, 0,GL_RGB, renderWidth, renderHeight, 0, GL_RGB, GL_UNSIGNED_BYTE, 0);

	// Poor filtering
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST); 
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

	// The depth buffer
    if (postDepthBuffer)
    {
        glDeleteRenderbuffers(1, &postDepthBuffer);
    }
	glGenRenderbuffers(1, &postDepthBuffer);
	glBindRenderbuffer(GL_RENDERBUFFER, postDepthBuffer);
	glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH_COMPONENT, renderWidth, renderHeight);
	glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, postDepthBuffer);

	// Set "renderedTexture" as our colour attachement #0
	glFramebufferTexture(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, postTexture, 0);

	// Set the list of draw buffers.
	postDrawBuffers[0] = GL_COLOR_ATTACHMENT0;
	glDrawBuffers(1, postDrawBuffers); // "1" is the size of DrawBuffers

	// Always check that our framebuffer is ok
	if (glCheckFramebufferStatus(GL_FRAMEBUFFER) != GL_FRAMEBUFFER_COMPLETE)
    {
        Log_Critical("Cannot generate render buffers\n");
        return;
    }

	glGenBuffers(1, &postQuadBufferVertex);
	glGenBuffers(1, &postQuadBufferUV);

    // prefill uv buffer, it stays
	glBindBuffer(GL_ARRAY_BUFFER, postQuadBufferUV);
    GLfloat uvBufferData[] = {
        0.0f, 0.0f,
        1.0f, 0.0f,
        0.0f, 1.0f,
        0.0f, 1.0f,
        1.0f, 0.0f,
        1.0f, 1.0f 
    };
	glBufferData(GL_ARRAY_BUFFER, sizeof(uvBufferData), uvBufferData, GL_STATIC_DRAW);

	// Create and compile our GLSL program from the shaders
    if (!postShaderId)
    {
        postShaderId = Shader::LoadShader("shaders/render.dsh")->GetProgram(0);
        if (postShaderId == 0)
        {
            Log_Critical("Post-processing shader wasn't loaded.\n");
        }
        postTextureId = glGetUniformLocation(postShaderId, "renderedTexture");
        currentPostShaderId = postShaderId;
    }
}

void RenderWindow::InitWindow(const char * title)
{
    if (current == NULL)
    {
        current = this;
    }

    SDL_GL_SetAttribute(SDL_GL_CONTEXT_MAJOR_VERSION, 2);
    SDL_GL_SetAttribute(SDL_GL_CONTEXT_MINOR_VERSION, 1);

    renderWidth = windowWidth = VIEWPORT_DEFAULT_WIDTH;
    renderHeight = windowHeight = VIEWPORT_DEFAULT_HEIGHT;
	window = SDL_CreateWindow(title, 100, 100, renderWidth, renderHeight, default_flags);
	assert(window);
	context = SDL_GL_CreateContext(window);

    if (context == NULL)
    {
        Log_Critical("OpenGL context could not be created! SDL Error: %s", SDL_GetError());
        return;
    }
    else
    {
        //Initialize GLEW
        glewExperimental = GL_TRUE; 
        GLuint glewError = glewInit();
        if (glewError != GLEW_OK)
        {
            Log_Critical("Error initializing GLEW! %s\n", (char *)glewGetErrorString(glewError));
        }

        //Use Vsync
        if (SDL_GL_SetSwapInterval(1) < 0)
        {
            Log_Critical("Warning: Unable to set VSync! SDL Error: %s\n", SDL_GetError());
        }
    }
    Clear();
}

void RenderWindow::Resize(u32 width, u32 height)
{
    windowWidth = width;
    windowHeight = height;
}

void RenderWindow::Clear() 
{
    u32 lastViewportWidth = renderWidth;
    u32 lastViewportHeight = renderHeight;
    if (displayMode == DisplayMode_Retro)
    {
        renderWidth = VIEWPORT_DEFAULT_WIDTH;
        renderHeight = VIEWPORT_DEFAULT_HEIGHT;
    }
    else
    {
        renderWidth = windowWidth;
        renderHeight = windowHeight;
    }
    if (lastViewportWidth != renderWidth || lastViewportHeight != renderHeight)
    {
        InitPostProcessing();
    }
    if (displayMode == DisplayMode_Fixed/*|| renderMode == RenderMode_Wireframe*/)
    {
        glBindFramebuffer(GL_FRAMEBUFFER, 0);
    }
    else
    {
        glBindFramebuffer(GL_FRAMEBUFFER, postFramebuffer);
    }
    glViewport(0, 0, renderWidth, renderHeight);
    Camera * camera = Camera::Current();
    if (camera != NULL)
    {
        glClearColor(camera->backgroundColor.x, camera->backgroundColor.y, camera->backgroundColor.z, 0.0f);
    } 
    else
    {
        glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
    }
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}

void RenderWindow::Prepare()
{
    Camera * cam = Camera::Current();
    if (cam != NULL)
    {
        float aspect = (float)renderWidth / (float)renderHeight;
        mtxProjection = glm::perspective(glm::radians((float)cam->fov), aspect, 0.1f, 1000.0f);
        mtxView = cam->GetViewMatrix();
    }
}

void RenderWindow::Present() 
{
    if (skipNext)
    {
        skipNext = false;
        return;
    }

    if (blankUntil > SDL_GetTicks())
    {
        Clear();
    }

    if (displayMode != DisplayMode_Fixed && renderMode != RenderMode_Wireframe)
    {
        // Render to the screen
        glBindFramebuffer(GL_FRAMEBUFFER, 0);
        glViewport(0, 0, windowWidth, windowHeight);
        glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

        // screen guad shader / texture
        glUseProgram(currentPostShaderId);
        glActiveTexture(GL_TEXTURE0);
        glBindTexture(GL_TEXTURE_2D, postTexture);
        glUniform1i(postTextureId, 0);

        // aspect ratio
        float ax = 1.0f;
        float ay = 1.0f;
        if (displayMode == DisplayMode_Retro)
        {
            float vRatio = (float)renderWidth / (float)renderHeight;
            float wRatio = (float)windowWidth / (float)windowHeight;
            if (vRatio < wRatio)
            {
                ax = vRatio / wRatio;
            }
            else
            {
                ay = wRatio / vRatio;
            }
        }

        // buffer verts
        glEnableVertexAttribArray(0);
        glBindBuffer(GL_ARRAY_BUFFER, postQuadBufferVertex);
        GLfloat quadBufferData[] = { 
            -ax, -ay, 0.0f,
            ax, -ay, 0.0f,
            -ax,  ay, 0.0f,
            -ax,  ay, 0.0f,
            ax, -ay, 0.0f,
            ax,  ay, 0.0f,
        };
        glBufferData(GL_ARRAY_BUFFER, sizeof(quadBufferData), quadBufferData, GL_STATIC_DRAW);
        glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, (void*)0);

        // buffer uvs
        glEnableVertexAttribArray(1);
        glBindBuffer(GL_ARRAY_BUFFER, postQuadBufferUV);
        glVertexAttribPointer(1, 2, GL_FLOAT, GL_FALSE, 0, (void*)0);

        // draw and clear
        glDrawArrays(GL_TRIANGLES, 0, 6);
        glDisableVertexAttribArray(0);
        glDisableVertexAttribArray(1);
    }

    SDL_GL_SwapWindow(window);
}

void RenderWindow::SkipNext()
{
    skipNext = true;
}

void RenderWindow::Blank(u32 milliseconds)
{
    blankUntil = SDL_GetTicks() + milliseconds;
}

void RenderWindow::SetPostShader(const char * shaderPath)
{
    if (shaderPath == NULL || shaderPath[0] == '\0')
    {
        currentPostShaderId = postShaderId;
    }
    else
    {
        currentPostShaderId = Shader::LoadShader(shaderPath)->GetProgram(0);
    }
}

void RenderWindow::SetFullScreen(bool value)
{
    isFullScreen = value == 1;
    if (value)
    {
        SDL_SetWindowFullscreen(window, default_flags | SDL_WINDOW_FULLSCREEN_DESKTOP);
        SDL_ShowCursor(SDL_DISABLE);
    }
    else
    {
        SDL_SetWindowFullscreen(window, default_flags);
        int ww, wh;
        SDL_GetWindowSize(window, &ww, &wh);
        Resize(ww, wh);
        SDL_ShowCursor(SDL_ENABLE);
    }
}

void RenderWindow::SetRenderMode(RenderMode value)
{
    renderMode = value;
    if (renderMode == RenderMode_Wireframe)
    {
        glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    }
    else
    {
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    }
    if (renderMode == RenderMode_Standard)
    {
        glEnable(GL_DEPTH_TEST);
    }
    else
    {
        glDisable(GL_DEPTH_TEST);
    }
}

void RenderWindow::RenderData(const GLfloat * vertices, u32 vertexCount, const GLfloat * uvs, u32 uvCount, const GLfloat * normals, u32 normalCount, const u32 * indices, u32 indexCount, const Material * material, const Matrix * matrix)
{
	Matrix model = matrix != NULL ? *matrix : Matrix(1.0f);
	Matrix mvp = mtxProjection * mtxView * model; 

    if (material != NULL)
    {
        material->Use();
    }
    else
    {

    }

    u32 bufferCount = 0;

	// This will identify our vertex buffer
	GLuint vertexBuffer;
	// Generate 1 buffer, put the resulting identifier in vertexbuffer
	glGenBuffers(1, &vertexBuffer);
	// The following commands will talk about our 'vertexbuffer' buffer
	glBindBuffer(GL_ARRAY_BUFFER, vertexBuffer);
	// Give our vertices to OpenGL.	
	glBufferData(GL_ARRAY_BUFFER, vertexCount * sizeof(GLfloat), vertices, GL_STATIC_DRAW);

	// 1st attribute buffer : vertices
	glEnableVertexAttribArray(bufferCount);
	glBindBuffer(GL_ARRAY_BUFFER, vertexBuffer);
	glVertexAttribPointer(
		0,                  // attribute 0. No particular reason for 0, but must match the layout in the shader.
		3,                  // size
		GL_FLOAT,           // type
		GL_FALSE,           // normalized?
		0,                  // stride
		(void*)0            // array buffer offset
	);
    bufferCount++;

    if (uvs != NULL && uvCount > 0)
    {
        GLuint uvBuffer;
        glGenBuffers(1, &uvBuffer);
        glBindBuffer(GL_ARRAY_BUFFER, uvBuffer);
        glBufferData(GL_ARRAY_BUFFER, uvCount * sizeof(GLfloat), &uvs[0], GL_STATIC_DRAW);
        glEnableVertexAttribArray(bufferCount);
        glBindBuffer(GL_ARRAY_BUFFER, uvBuffer);
        glVertexAttribPointer(
            1,                                // attribute. No particular reason for 1, but must match the layout in the shader.
            2,                                // size : U+V => 2
            GL_FLOAT,                         // type
            GL_FALSE,                         // normalized?
            0,                                // stride
            (void*)0                          // array buffer offset
        );
        bufferCount++;
    }

    if (normals != NULL && normalCount > 0)
    {
        GLuint normalBuffer;
        glGenBuffers(1, &normalBuffer);
        glBindBuffer(GL_ARRAY_BUFFER, normalBuffer);
        glBufferData(GL_ARRAY_BUFFER, normalCount * sizeof(GLfloat), &normals[0], GL_STATIC_DRAW);
        glEnableVertexAttribArray(bufferCount);
        glBindBuffer(GL_ARRAY_BUFFER, normalBuffer);
        glVertexAttribPointer(
            2,                                // attribute
            3,                                // size
            GL_FLOAT,                         // type
            GL_FALSE,                         // normalized?
            0,                                // stride
            (void*)0                          // array buffer offset
        );
        bufferCount++;
    }

    if (indices != NULL && indexCount > 0)
    {
        GLuint elementBuffer;
        glGenBuffers(1, &elementBuffer);
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, elementBuffer);
        glBufferData(GL_ELEMENT_ARRAY_BUFFER, indexCount * sizeof(u32), &indices[0], GL_STATIC_DRAW);
        
        // Index buffer
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, elementBuffer);

        // Draw the triangles !
        glDrawElements(
            GL_TRIANGLES,      // mode
            indexCount,    // count
            GL_UNSIGNED_INT,   // type
            (void*)0           // element array buffer offset
        );
    }
    else
    {
        glDrawArrays(GL_TRIANGLES, 0, vertexCount);
    }

    for (u32 i=0; i<bufferCount; i++)
    {
	    glDisableVertexAttribArray(i);	
    }
}

void RenderWindow::RenderMesh(const Mesh * mesh, const Matrix * matrix)
{
    const std::vector<Vector3> & vertices = mesh->GetVertices();
    const std::vector<Vector2> & uvs = mesh->GetUvs();
    const std::vector<Vector3> & normals = mesh->GetNormals();
    const std::vector<u32> & indices = mesh->GetIndices();
    RenderData(
        (GLfloat*)&vertices[0], Mesh::ClampBuffer(vertices.size() * 3), 
        (GLfloat*)&uvs[0], Mesh::ClampBuffer(uvs.size() * 2),
        (GLfloat*)&normals[0], Mesh::ClampBuffer(normals.size() * 3),
        (u32*)&indices[0], Mesh::ClampBuffer(indices.size()),
        mesh->GetMaterial().Inst(),
        matrix
    );
}

RenderWindow * RenderWindow::Current()
{
    return current;
}