#include <Model3D.h>
#include <read/iqm/IqmReader.h>
#include <FileSystem.h>
#include <string.h>
#include <Cache.h>
#include <MeshRenderer.h>
#include <Maths.h>

CacheMap<ModelReader> readerCache;

Model3D::Model3D()
{
    mesh = new Mesh();
    material = NULL;
    reader = NULL;
}

Model3D::~Model3D()
{
    Material::Free(material);
    if (renderer != NULL)
    {
        delete renderer;
        renderer = NULL;
    }
    readerCache.Free(reader);
}

ModelFormat Model3D::GetModelFormat(const char * modelPath)
{
    if (FileSystem::IsExtenstion(modelPath, "iqm"))
    {
        return ModelFormat_IQM;
    }
    return ModelFormat_NONE;
}


bool Model3D::LoadModel(const char * modelPath, float scale)
{
    return LoadModel(modelPath, modelPath, scale);
}

bool Model3D::LoadModel(const char * modelPath, const char * animationsPath, float scale)
{
    isLoaded = false;
    modelHash = Hash(modelPath) + Hash(animationsPath);
    format = GetModelFormat(modelPath);
    transform.scale = Vector3(scale);
    reader = readerCache.Get(modelHash);
    if (reader == NULL)
    {
        ModelReader * newReader = NULL;
        switch (format)
        {
            case ModelFormat_IQM:
                newReader = new IqmReader();
                break;
            default:
            {
                Log_Error("Can't read file format: %s", modelPath);
                return false;
            }
        }

        std::string fullPath = FileSystem::GetAbsolutePath(modelPath);
        if (newReader->Load(fullPath.c_str()) == false)
        {
            Log_Error("Can't read model: %s", fullPath.c_str());
            return false;
        }

        if (format == ModelFormat_IQM && HashEqual(modelPath, animationsPath) == false)
        {
            fullPath = FileSystem::GetAbsolutePath(animationsPath);
            ((IqmReader *)newReader)->LoadAnimations(fullPath.c_str());
        }

        readerCache.Set(modelHash, newReader);
        reader = readerCache.Get(modelHash);
    }

    Update(0.0f);
    isLoaded = true;
    return true;
}

void Model3D::Clear()
{
    isLoaded = false;
}

void Model3D::SetMaterial(Ref<Material> & value)
{
    material = value;
    mesh->material = material;
}

void Model3D::SetMaterial(const char * texturePath, const char * shaderPath)
{
    material = Material::Load(texturePath, shaderPath);
    mesh->material = material;
}

void Model3D::BlendAnimation(ModelAnimData * anim, float blendTime)
{
    if (anim != animCurrent)
    {
        animPrevious = animCurrent;
        animCurrent = anim;
        blendDuration = blendTime;
        blendRemaining = blendTime;
    }
}

void Model3D::BlendAnimation(ModelAnimData * anim, float blendTime, float _fps)
{
    fps = _fps;
    BlendAnimation(anim, blendTime);
}

void Model3D::PlayAnimation(ModelAnimData * anim, float _time)
{
    if (anim != animCurrent)
    {
        animPrevious = animCurrent;
        animCurrent = anim;
        blendDuration = 0.0f;
        blendRemaining = 0.0f;
        time = _time;
    }
}

void Model3D::PlayAnimation(ModelAnimData * anim, float _time, float _fps)
{
    fps = _fps;
    PlayAnimation(anim, _time);
}

ModelAnimData * Model3D::FindAnimation(const char * animName) const 
{
    if (reader.Inst() == NULL)
        return NULL;
    return reader.Inst()->FindAnimation(animName);
}

const ModelBoneData * Model3D::FindBone(const char * boneName) const 
{
    if (reader.Inst() == NULL)
        return NULL;
    return reader.Inst()->FindBone(boneName);
}

Matrix Model3D::GetBonePose(const ModelBoneData * bone, bool scaled) const
{
    Matrix result = transform.GetMatrix();
    if (bone != NULL)
    {
        result *= bone->transform.GetMatrix();
        if (scaled == false)
        {
            if (Maths::IsWithinDistance(transform.scale, 0.001f) == false)
            {
                result = glm::scale(result, Vector3(1.0f / transform.scale.x, 1.0f / transform.scale.y, 1.0f / transform.scale.z));
            }
        }
    }
    return result;
}

void Model3D::Update(float dt)
{
    time += dt * fps * MODEL_FPS_MULTIPLIER;
    if (reader != NULL)
    {
        float blend = 0.0f;
        if (animPrevious != NULL)
        {
            if (blendRemaining > 0.0f)
            {
                blendRemaining -= dt;
            }
            if (blendRemaining < 0.0f)
            {
                blendRemaining = 0.0f;
            }
            blend = blendRemaining / (blendDuration > 0.0f ? blendDuration : 1.0f);
        }
        reader->Process(mesh, &time, animCurrent, animPrevious, blend);
        time = fmodf(time, reader->GetFrameCount());
    }
}

void Model3D::Render(MeshRenderer * renderer, Matrix matrix)
{
    mtxApplied = matrix * transform.GetMatrix();
    renderer->Render(mesh, &mtxApplied);
}

void Model3D::Render(Matrix matrix)
{
    if (renderer == NULL)
    {
        renderer = new MeshRenderer();
    }
    Render(renderer, matrix);
}
