#include "ModMusicPlayer.h"
#include <desslibs.h>
#include <libmodplug.h>
#include <FileSystem.h>

ModMusicPlayer * ModMusicPlayer::current = NULL;

ModMusicPlayer * ModMusicPlayer::Current()
{
    return current;
}

void ModMusicPlayer::AudioCallback(void * userdata, u8 * stream, int len)
{
    if (current != NULL)
    {
        current->UpdateAudio(stream, len);
    }
}

ModMusicPlayer::ModMusicPlayer()
{
    current = this;
    if (Init() == false)
    {
        isMuted = true;
    }
}

bool ModMusicPlayer::Init()
{
    if (SDL_Init(SDL_INIT_AUDIO) < 0)
    {
        Log_Critical("Couldn't init audio: %s\n", SDL_GetError());
        return false;
    }
    return true;
}

ModMusicPlayer::~ModMusicPlayer()
{
    Unload();
}

bool ModMusicPlayer::PlayPatterns(u16 start, u16 end, u16 repeat)
{
    StopOverride();
    ModMusicTrack & track = tracks[MusicTrack_Main];
    if (track.file == NULL)
    {
        track.file = &files[0];
    }
    track.cue.start = start;
    track.cue.end = end;
    track.cue.repeat = repeat;
    return Play(&track.cue);
}

const ModMusicCue * ModMusicPlayer::GetCue(const char * id)
{
    const ModMusicCue * result = GetCue(Hash(id));
    if (result == NULL)
    {
        Log_Error("Can't find cue %s\n", id);
    }
    return result;
}

const ModMusicCue * ModMusicPlayer::GetCue(HashInt cueHash)
{
    for (std::vector<ModMusicCue>::iterator it = cues.begin(); it != cues.end(); ++it)
    {
        if (it->hash == cueHash)
        {
            return &(*it);
            break;
        }
    }
    Log_Error("Can't find cue %i\n", cueHash);
    return NULL;
}

bool ModMusicPlayer::HasCue(const char * id) const
{
    return HasCue(Hash(id));
}

bool ModMusicPlayer::HasCue(HashInt cueHash) const
{
    if (cueHash > 0)
    {
        for (std::vector<ModMusicCue>::const_iterator it = cues.begin(); it != cues.end(); ++it)
        {
            if (it->hash == cueHash)
            {
                return true;
            }
        }
    }
    return false;
}

bool ModMusicPlayer::Play(const char * cueName)
{
    return Play(GetCue(cueName));
}

bool ModMusicPlayer::Play(const ModMusicCue * cue)
{
    if (isMuted)
        return false;
    if (cue == NULL)
    {
        Log_Error("Can't play null cue\n");
        return false;
    }
    StopOverride();
    bool isContinuous = IsContinuous(MusicTrack_Main, cue);
    if (LoadCueToTrack(MusicTrack_Main, cue) == false)
    {
        return false;
    }
    if (isContinuous == false)
    {
        tracks[MusicTrack_Main].pattern = tracks[MusicTrack_Main].cue.start;
        ModPlug_SeekOrder(tracks[MusicTrack_Main].file->mod, tracks[MusicTrack_Main].pattern);
        tracks[MusicTrack_Main].position = 0;
    }
    isPlaying = true;
    return true;
}

bool ModMusicPlayer::IsContinuous(ModMusicTrackIndex track, const ModMusicCue * cue) const
{
    if (tracks[track].file == NULL)
        return false;

    if (cue->fileHash != tracks[track].file->hash)
        return false;

    return cue->start >= tracks[track].cue.start && cue->start < tracks[track].cue.end;
}

bool ModMusicPlayer::IsPersistant() const
{
    if (tracks[MusicTrack_Main].file == NULL)
        return false;

    return tracks[MusicTrack_Main].cue.persistant;
}

bool ModMusicPlayer::PlayOverride(const ModMusicCue * cue)
{
    if (tracks[MusicTrack_Main].file != NULL && tracks[MusicTrack_Main].cue.persistant)
    {
        return false;
    }

    if (
        tracks[MusicTrack_Override].file != NULL && 
        tracks[MusicTrack_Override].pattern >= cue->start && 
        tracks[MusicTrack_Override].pattern <= cue->end
    )
    {
        isPlaying = true;
        return true;
    }
    if (LoadCueToTrack(MusicTrack_Override, cue) == false)
    {
        return false;
    }
    ModPlug_SeekOrder(tracks[MusicTrack_Override].file->mod, tracks[MusicTrack_Override].cue.start);
    isPlaying = true;
    return true;
}

bool ModMusicPlayer::LoadCueToTrack(ModMusicTrackIndex slot, const ModMusicCue * cue)
{
    if (IsContinuous(slot, cue) == false)
    {
        if (SetFile(slot, cue->fileHash) == false)
        {
            Log_Error("Can't play music cue - missing mod.\n");
            return false;
        }
    }
    tracks[slot].cue = ModMusicCue(*cue);
    tracks[slot].cue.hash = cue->hash;
    tracks[slot].cue.start = cue->start;
    tracks[slot].cue.end = cue->end;
    tracks[slot].cue.repeat = cue->repeat != 0 ? cue->repeat : cue->start;
	ModPlug_SetMasterVolume(tracks[slot].file->mod, 480);
    SDL_PauseAudioDevice(tracks[slot].file->deviceId, 0);
    return true;
}

bool ModMusicPlayer::SetFile(ModMusicTrackIndex slot, const char * modPath)
{
    tracks[slot].file = LoadMod(modPath);
    if (tracks[slot].file == NULL)
    {
        Log_Error("Can't play file %s\n", modPath);
        return false;
    }
    return true;
}

void ModMusicPlayer::SetMuted(bool value)
{
    isMuted = value;
    if (value)
    {
        Pause();
    }
    else
    {
        Resume();
    }
}

bool ModMusicPlayer::SetFile(ModMusicTrackIndex slot, HashInt fileHash)
{
    for (std::vector<ModMusicFile>::iterator it = files.begin(); it != files.end(); ++it)
    {
        if (it->hash == fileHash)
        {
            tracks[slot].file = &(*it);
            return true;
        }
    }
    return false;
}

HashInt ModMusicPlayer::GetCue() const
{
    return tracks[MusicTrack_Main].cue.hash;
}

u16 ModMusicPlayer::GetPattern() const
{
    return ModPlug_GetCurrentPattern(tracks[MusicTrack_Main].file->mod);
}

void ModMusicPlayer::UpdateAudio(u8 * stream, int len)
{
    ModMusicTrack * track = &tracks[MusicTrack_Override];
    if (track->file == NULL)
    {
        track = &tracks[MusicTrack_Main];
    }
    if (track->file == NULL)
    {
        return;
    }

    int bytesRead = ModPlug_Read(track->file->mod, stream, len);
    track->position += bytesRead;
    track->pattern = ModPlug_GetCurrentOrder(track->file->mod);
    if (track->cue.end > 0 && track->pattern > track->cue.end)
    {
        track->beat = ModPlug_GetCurrentRow(track->file->mod);
        u32 samplesPerBeat = track->file->spec->freq * 60 / ModPlug_GetCurrentTempo(track->file->mod);
        u32 bytesPerBeat = samplesPerBeat * track->file->spec->channels / 2;
        u32 bytesOver = (track->position % bytesPerBeat) + track->beat * bytesPerBeat;
        u32 writeIndex = (len - bytesOver) % len;
        u16 repeat = track->cue.repeat == 0 ? track->cue.start : track->cue.repeat;
        ModPlug_SeekOrder(track->file->mod, repeat);
        track->position = ModPlug_Read(track->file->mod, stream + writeIndex, len - writeIndex);    
    }
}

void ModMusicPlayer::Update(float dt)
{
    if (pauseTime > 0.0f)
    {
        pauseTime -= dt;
        if (pauseTime <= 0.0f)
        {
            Resume();
        }
    }
}

void ModMusicPlayer::Pause()
{
    for (i8 i=MusicTrack_COUNT-1; i>=0; i--)
    {
        if (tracks[i].file != NULL)
        {
            SDL_PauseAudioDevice(tracks[i].file->deviceId, 1);
            return;
        }
    }
    isPlaying = false;
}

void ModMusicPlayer::PauseFor(float time)
{
    pauseTime = time;
    Pause();
    for (i8 i=MusicTrack_COUNT-1; i>=0; i--)
    {
        if (tracks[i].file != NULL)
        {
            pausePos = ModPlug_GetCurrentOrder(tracks[MusicTrack_Main].file->mod);
            return;
        }
    }
    pausePos = 0;
}

void ModMusicPlayer::Resume()
{
    for (i8 i=MusicTrack_COUNT-1; i>=0; i--)
    {
        if (tracks[i].file != NULL)
        {
            SDL_PauseAudioDevice(tracks[i].file->deviceId, 0);
            isPlaying = true;
            ModPlug_SeekOrder(tracks[i].file->mod, pausePos);
            return;
        }
    }
}

void ModMusicPlayer::StopOverride()
{
    if (tracks[MusicTrack_Override].file != NULL)
    {
        SDL_PauseAudioDevice(tracks[MusicTrack_Override].file->deviceId, 1);
        tracks[MusicTrack_Override].file = NULL;
        if (tracks[MusicTrack_Main].file != NULL)
        {
            SDL_PauseAudioDevice(tracks[MusicTrack_Main].file->deviceId, 0);
            ModPlug_SeekOrder(tracks[MusicTrack_Main].file->mod, tracks[MusicTrack_Main].pattern);
        }
    }
}

void ModMusicPlayer::Stop()
{
    pauseTime = 0.0f;
    Pause();
    StopOverride();
    for (i8 i=MusicTrack_COUNT-1; i>=0; i--)
    {
        if (tracks[i].file != NULL)
        {
            SDL_PauseAudioDevice(tracks[i].file->deviceId, 1);
            tracks[i].file = NULL;
        }
    }
}

ModMusicFile * ModMusicPlayer::LoadMod(const char * modPath)
{
    HashInt hash = Hash(modPath);
    for (std::vector<ModMusicFile>::iterator it = files.begin(); it != files.end(); ++it)
    {
        if (it->hash == hash)
        {
            return &(*it);
        }
    }

    SDL_RWops * file = FileSystem::FileOpen(modPath, "r+b");
	if (file == NULL)
	{
		Log_Error("File cannot be read: %s\n", modPath);
		return NULL;
	}
    
    u32 fileSize = (u32)SDL_RWsize(file);
    void * fileBuffer = malloc(fileSize);
    if (FileSystem::FileRead(file, fileBuffer, fileSize, 1) == 0)
    {
        Log_Error("Mod read error cannot be read: %s: %s\n", modPath, SDL_GetError());
        return NULL;
    }
    FileSystem::FileClose(file);

    SDL_AudioSpec request;
    request.freq = 44100;
    request.channels = 2;
    request.format = AUDIO_S16SYS;
    request.callback = &AudioCallback;
    
    files.push_back(ModMusicFile());
    ModMusicFile & modFile = files.back();
    modFile.hash = hash;
    modFile.spec = new SDL_AudioSpec();
    modFile.deviceId = SDL_OpenAudioDevice(NULL, 0, &request, modFile.spec, 0);
    modFile.mod = ModPlug_Load(fileBuffer, fileSize);
    return &modFile;
}

void ModMusicPlayer::Unload()
{
}
