#include <Mesh.h>
#include <string.h>

u32 Mesh::ClampBuffer(size_t n)
{
    if (n > UINT16_MAX)
    {
        n = UINT16_MAX;
    }
    return (u32)n;
}

template <typename T>
void SetToVector(const T * arr, u32 count, std::vector<T> & out, size_t startIndex = UINT32_MAX)
{
    if (startIndex == UINT32_MAX)
    {
        startIndex = out.size();
    }
    u32 newCount = Mesh::ClampBuffer(startIndex + count);
    if (out.size() != newCount)
    {
        out.resize(newCount);
    }
    for (size_t i = 0; i < count; i++)
    {
        out[startIndex + i] = arr[i];
    }
}

template <typename T>
void SetToVector(const std::vector<T> & in, std::vector<T> & out)
{
    u32 inSize = Mesh::ClampBuffer(in.size());
    u32 outSize = Mesh::ClampBuffer(out.size());
    if (outSize != inSize)
    {
        out.resize(inSize);
    }
    for (u32 i = 0; i < outSize; i++)
    {
        out[i] = in[i];
    }
}

void NormalizeVectors(std::vector<Vector3>& normals)
{
    for (size_t i = 0, lim = normals.size(); i < lim; i++)
    {
        if (normals[i].x == 0.0f && normals[i].y == 0.0f && normals[i].z == 0.0f)
        {
            normals[i] = Vector3_UP;
        }
    }
}

void CopyUvs(const std::vector<Vector2> & in, std::vector<Vector2> & out, bool flip = false)
{
    u32 newSize = Mesh::ClampBuffer(in.size());
    if (out.size() != newSize)
    {
        out.resize(newSize);
        for (u32 i = 0; i < newSize; i++)
        {
            out[i] = Vector2(in[i].x, flip ? 1.0f - in[i].y : in[i].y);
        }
    }
}

Mesh::Mesh() :
    material(NULL),
    vertices(),
    uvs(),
    colors(),
    indices(),
    normals()
{}

Mesh::Mesh(const Mesh & mesh)
{
    Copy((Mesh*)&mesh);
}

Mesh::~Mesh()
{
    Clear();
}

void Mesh::Clear()
{
    Material::Free(material);
    ClearGeometry();
}

void Mesh::ClearGeometry()
{
    vertices.clear();
    uvs.clear();
    colors.clear();
    indices.clear();
    normals.clear();
}

void Mesh::Copy(const Mesh * other)
{
    SetToVector(other->vertices, vertices);
    SetToVector(other->uvs, uvs);
    SetToVector(other->colors, colors);
    SetToVector(other->normals, normals);
    SetToVector(other->indices, indices);
    NormalizeVectors(normals);
    if (other->material.IsValid())
    {
        material = Material::Get(other->GetMaterial());
    }
}

void Mesh::SetMaterial(Ref<Material> & value)
{
    Material::Free(material);
    material = value;
}

void Mesh::SetMaterial(Material * value)
{
    material = Ref<Material>(value);
}

void Mesh::AddVertices(Vector3 * arr, u32 count)
{
    SetToVector(arr, count, vertices);
}
void Mesh::AddVertices(std::vector<Vector3> & source)
{
    AddVertices(&source[0], ClampBuffer(source.size()));
}

void Mesh::AddUvs(Vector2 * arr, u32 count)
{
    SetToVector(arr, count, uvs);
}
void Mesh::AddUvs(std::vector<Vector2> & source)
{
    AddUvs(&source[0], ClampBuffer(source.size()));
}

void Mesh::AddNormals(Vector3 * arr, u32 count)
{
    SetToVector(arr, count, normals);
    NormalizeVectors(normals);
}
void Mesh::AddNormals(std::vector<Vector3> & source)
{
    AddNormals(&source[0], ClampBuffer(source.size()));
}

void Mesh::AddColors(Vector3 * arr, u32 count)
{
    SetToVector(arr, count, colors);
}
void Mesh::AddColors(std::vector<Vector3> & source)
{
    AddColors(&source[0], ClampBuffer(source.size()));
}

void Mesh::AddIndices(u32 * arr, u32 count)
{
    SetToVector(arr, count, indices);
}
void Mesh::AddIndices(std::vector<u32> & indices)
{
    AddIndices(&indices[0], ClampBuffer(indices.size()));
}

void Mesh::AddTriangleFan(std::vector<Vector3> & _vertices, std::vector<Vector2> & _uvs, std::vector<Vector3> & _normals)
{
    u32 vertexCount = ClampBuffer(vertices.size());
    u32 appendVertexCount = ClampBuffer(_vertices.size());
    for (u32 i=0; i<appendVertexCount - 2; i++)
    {
        indices.push_back(vertexCount);
        indices.push_back(vertexCount + i + 1);
        indices.push_back(vertexCount + i + 2);
    }
    uvs.resize(vertexCount);
    normals.resize(vertexCount);
    SetToVector(&_vertices[0], appendVertexCount, vertices, vertexCount);
    SetToVector(&_uvs[0], appendVertexCount, uvs, vertexCount);
    SetToVector(&_normals[0], appendVertexCount, normals, vertexCount);
    NormalizeVectors(normals);
}

void Mesh::AddTriangleFan(std::vector<Vector3> & _vertices)
{
    u32 firstI = ClampBuffer(vertices.size());
    u32 newVertexCount = ClampBuffer(_vertices.size());
    for (u32 i=0; i<newVertexCount - 2; i++)
    {
        indices.push_back(firstI);
        indices.push_back(firstI + i + 1);
        indices.push_back(firstI + i + 2);
    }
    AddVertices(_vertices);
    NormalizeVectors(normals);
}

void Mesh::SetVertexCount(u16 count)
{
    if (vertices.size() != count)
    {
        vertices.resize(count);
        uvs.resize(count);
        normals.resize(count);
        for (u16 i = 0; i < count; i++)
        {
            vertices[i] = Vector3_ZERO;
            uvs[i] = Vector2_ZERO;
            normals[i] = Vector3_UP;
        }
    }
}

void Mesh::SetIndexCount(u16 count)
{
    if (indices.size() != count)
    {
        indices.resize(count);
    }
}