#ifndef _DESS_MATHS_H
#define _DESS_MATHS_H

#include <desstypes.h>
#include <glm/gtx/quaternion.hpp>
#include <glm/ext/quaternion_transform.hpp>
#include <random>
    
#define DELTA_EASE_FRAME_MAGNIFY 60.0f

class Maths
{
public:
    static float Lerp(float a, float b, float t)
    {
        t = Clamp01(t);
        return a * (1.0f - t) + b * t;
    }
    
    static Vector3 Lerp(Vector3 a, Vector3 b, float t)
    {
        a.x = Lerp(a.x, b.x, t);
        a.y = Lerp(a.y, b.y, t);
        a.z = Lerp(a.z, b.z, t);
        return a;
    }

    static Quat Slerp(Quat a, Quat b, float t)
    {
        return glm::slerp(a, b, t);
    }
    
    static float DeltaEase(float sharpness, float dt)
    {
        float result = Clamp01(1.0f - Pow(1.0f - sharpness, dt * DELTA_EASE_FRAME_MAGNIFY));
        return result;
    }

    static Vector3 Lerp(Vector3 a, Vector3 b, float sharpness, float dt)
    {
        return Lerp(a, b, DeltaEase(sharpness, dt));
    }

    static float Lerp(float a, float b, float sharpness, float dt)
    {
        return Lerp(a, b, DeltaEase(sharpness, dt));
    }

    static Quat Slerp(Quat a, Quat b, float sharpness, float dt)
    {
        return Slerp(a, b, DeltaEase(sharpness, dt));
    }

    static float Length(Vector3 v)
    {
        return sqrtf(LengthSq(v));
    }

    static float Length(Vector2 v)
    {
        return sqrtf(LengthSq(v));
    }

    static float LengthSq(Vector3 v)
    {
        if (IsNaN(v))
            return 0.0f;
        float result = v.x * v.x + v.y * v.y + v.z * v.z;
        if (isfinite(result))
            return result;
        return 0.0f;
    }

    static bool IsWithinDistance(Vector3 a, Vector3 b, float dist)
    {
        return LengthSq(a - b) < dist * dist;
    }

    static bool IsWithinDistance(Vector3 v, float dist)
    {
        return LengthSq(v) < dist * dist;
    }
    
    static float LengthSq(Vector2 v)
    {
        return v.x * v.x + v.y * v.y;
    }

    static float Pow(float n, float m)
    {
        return pow(n, m);
    }

    static Vector3 Clamp(Vector3 v, float length)
    {
        if (LengthSq(v) > length * length)
        {
            v = Normalize(v) * length;
        }
        return v;
    }

    static Vector2 Clamp(Vector2 v, float length)
    {
        if (LengthSq(v) > length * length)
        {
            v = Normalize(v) * length;
        }
        return v;
    }

    static Vector3 Normalize(Vector3 v, Vector3 fallback = Vector3_UP)
    {
        Vector3 result = glm::normalize(v);
        if (IsNaN(result))
            return fallback;
        return result;
    }

    static Vector2 Normalize(Vector2 v, Vector2 fallback = Vector2_ZERO)
    {
        Vector2 result = glm::normalize(v);
        if (IsNaN(result))
            return fallback;
        return result;
    }

    inline static float Abs(float a)
    {
        return a < 0.0f ? -a : a;
    }

    inline static i32 Abs(i32 a)
    {
        return a < 0 ? -a : a;
    }

    inline static i32 Sign(i32 x)
    {
        return x < 0 ? -1 : x > 0 ? 1 : 0;
    }

    inline static float Sign(float x)
    {
        return x < 0.0f ? -1.0f : x > 0.0f ? 1.0f : 0.0f;
    }

    static Vector3 Cardinal(Vector3 v)
    {
        if (Abs(v.x) > Abs(v.z))
        {
            return Vector3(v.x > 0.0f ? 1.0f : -1.0f, 0, 0);
        }
        else if (v.z != 0.0f)
        {
            return Vector3(0, 0, v.z > 0.0f ? 1.0f : -1.0f);
        }
        return Vector3_ZERO; 
    }

    inline static Vector3 Cross(Vector3 a, Vector3 b)
    {
        return glm::cross(a, b);
    }

    static float Dot(Vector3 a, Vector3 b)
    {
        return glm::dot(a, b);
    }

    inline static bool IsNaN(float n)
    {
        return isnan(n);
    }

    inline static bool IsNaN(Vector2 v)
    {
        return IsNaN(v.x) || IsNaN(v.y);
    }

    inline static bool IsNaN(Vector3 v)
    {
        return IsNaN(v.x) || IsNaN(v.y) || IsNaN(v.z);
    }

    static Quat LookRotation(Vector3 direction, Vector3 up = Vector3_UP)
    {
        direction = Normalize(direction, Vector3_FORWARD);
        return glm::quatLookAt(direction, up);
    }

    static float Clamp(float n, float min, float max)
    {
        return n < min ? min : n > max ? max : n;
    }

    static float Clamp(float n, float limit)
    {
        return Clamp(n, -Abs(limit), Abs(limit));
    }

    static float Clamp01(float n)
    {
        return Clamp(n, 0.0f, 1.0f);
    }

    static inline float Max(float a, float b)
    {
        return a > b ? a : b;
    }

    static inline float Min(float a, float b)
    {
        return a < b ? a : b;
    }

    static int RandI(int max)
    {
        return rand() % max;
    }

    static int RandI(int min, int max)
    {
        return (rand() % (max - min)) + min;
    }

    static float RandF()
    {
        return (float)(rand()) / (float)(RAND_MAX);
    }

    static float RandF(float max)
    {
        return RandF() * max;
    }

    static float RandF(float min, float max)
    {
        return RandF() * (max - min) + min;
    }

    static float RandFSigned(float max)
    {
        return RandF(-max, max);
    }
    
    static float RandFMultiplier(float value, float m)
    {
        return value - m + RandF(m + m);
    }
};

#endif