#include <Material.h>
#include <Texture.h>
#include <Shader.h>
#include <desslibs.h>
#include <Hash.h>
#include <Cache.h>

const int MATERIALS_OWNER = 1; 
const Material * MATERIAL_NONE = NULL;

Ref<Material> & Material::Load(const char * texturePath, const char * shaderPathPrefix)
{
    HashInt hash = Hash(texturePath);
    if (shaderPathPrefix == NULL)
    {
        shaderPathPrefix = MATERIAL_DEFAULT_PATH;
    }
    hash += Hash(shaderPathPrefix);
    Ref<Material> & existing = GetCache<Material>().Get(hash);
    if (existing.IsValid())
    {
        return existing;
    }
    Material * newMat = new Material();
    newMat->owner = (void*)&MATERIALS_OWNER;
    newMat->LoadTexture(texturePath, shaderPathPrefix);
    newMat->hash = hash;
    GetCache<Material>().Set(hash, newMat);
    return GetCache<Material>().Get(hash);
}

Ref<Material> & Material::Get(const Ref<Material> & material)
{
    return GetCache<Material>().Get(material);
}

Ref<Material> & Material::Clone(const Ref<Material> & material)
{
    if (material.IsValid() == false)
    {
        Log_Error("Invalid source material\n");
        return NoRef<Material>();
    }
    Material * newMat = new Material(material.Inst());
    newMat->owner = (void*)&MATERIALS_OWNER;
    HashInt hash = GetCache<Material>().GetNextFreeHash(material);
    GetCache<Material>().Set(hash, newMat);
    return GetCache<Material>().Get(hash);
}

void Material::Free(Ref<Material> & material)
{
    GetCache<Material>().Free(material);
}

Material::Material()
{

}

Material::Material(const Material * original)
{
    Copy(original);
}

void Material::Copy(const Material * other)
{
    if (other != NULL)
    {
        shaderHash = other->shaderHash;
        textureHash = other->textureHash;
        color = other->color;
        shader = other->shader;
        textureId = other->textureId;
    }
}

void Material::LoadShader(const char * shaderPathPrefix)
{
    HashInt newHash = Hash(shaderPathPrefix);
    if (newHash != shaderHash)
    {
        shader = Shader::LoadShader(shaderPathPrefix);
        shaderHash = newHash;
    }
    hash = textureHash + shaderHash;
}

void Material::SetShader(const Shader * value)
{
    shader = value;
    shaderHash = 0;
    hash = textureHash;
}

void Material::LoadTexture(const char * texturePath, const char * shaderPathPrefix)
{
    if (shaderPathPrefix != NULL)
    {
        LoadShader(shaderPathPrefix);
    }
    HashInt newHash = Hash(texturePath);
    if (newHash != textureHash)
    {
        textureId = Texture::LoadTexture(texturePath);
        textureHash = newHash;
    }
    hash = textureHash + shaderHash;
}

const ShaderPass * Material::Use(u8 passIndex) const
{
    if (shader == NULL)
        return NULL;

    const ShaderPass * program = shader->GetPass(passIndex);
    if (program == NULL)
        return NULL;

	glUseProgram(program->programId); 

    if (textureId != 0 && program->uniformSAMPLER >= 0)
    {
        glActiveTexture(GL_TEXTURE0);
        glBindTexture(GL_TEXTURE_2D, textureId);
        glUniform1i(program->uniformSAMPLER, 0);
    }

    return program;
}