#ifndef _DESS_INPUT_H
#define _DESS_INPUT_H

#include <desstypes.h>
#include "SDL2/SDL.h"

#define INPUT_GAMEPAD_COUNT 1
#define KEYBOARD_BUFFER_SIZE 8

enum InputButtonState
{
    InputButtonState_OFF,
    InputButtonState_RELEASED,
    InputButtonState_PRESSED,
    InputButtonState_HELD
};

enum InputButton
{
    InputButton_NONE,
    InputButton_A,
    InputButton_B,
    InputButton_START,
    InputButton_BACK,
    InputButton_COUNT
};

enum InputStick
{
    InputStick_LEFT,
    InputStick_COUNT
};

enum InputAxis
{
    InputAxis_H,
    InputAxis_V
};

struct InputKeyboardKey
{
    u16 scancode = 0;
    char key = '\0';
    InputButtonState state = InputButtonState_OFF;
};

struct InputKeyboardBinding
{
    u16 key;
    u16 alt;
};

class InputKeyboardMap
{
public:
    u8 gamepadIndex = 0;
    u16 moveLeft[2];
    u16 moveRight[2];
    u16 moveUp[2];
    u16 moveDown[2];
    u16 viewLeft[2];
    u16 viewRight[2];
    u16 viewUp[2];
    u16 viewDown[2];
    InputKeyboardBinding buttons[InputButton_COUNT];
    u16 end;
    float axisValue[4];

    InputKeyboardMap() 
    {
        Unbind();
    }

    void Unbind()
    {
        for (u16 * it = &moveLeft[0]; it != &end; ++it)
        {
            *it = 0;
        }
        for (u8 i = 0; i < 4; i++)
        {
            axisValue[i] = 0.0f;
        }
    }

    void SetButtonBinding(InputButton button, u16 key, u16 alt = 0)
    {
        buttons[button].key = key;
        buttons[button].alt = alt;
    }

    InputKeyboardBinding GetButtonBinding(InputButton button)
    {
        return buttons[button];
    }
};

class InputGamepad
{
public:
    i32 joytickId = 0;
    float stickLeft[2] = { 0.0f, 0.0f };
    InputButtonState buttons[InputButton_COUNT];

    InputGamepad()
    {
        Reset(); 
    }
    
    bool IsHeld(InputButton button)
    {
        return buttons[button] >= InputButtonState_PRESSED;
    }

    bool WasPressed(InputButton button)
    {
        return buttons[button] == InputButtonState_PRESSED;
    }

    Vector2 GetStick(InputStick stick)
    {
        return Vector2(stickLeft[0], stickLeft[1]);
    }

    void SetStick(InputStick stick, InputAxis axis, float value)
    {
        stickLeft[axis] = value;
    }

    void SetStick(InputStick stick, float x, float y)
    {   
        stickLeft[0] = x;
        stickLeft[1] = y;
    }

    void Clamp()
    {
        for (u8 i=0; i<2; i++)
        {
            stickLeft[i] = stickLeft[i] < -1.0f ? -1.0f : stickLeft[i] > 1.0f ? 1.0f : stickLeft[i];
        }
    }

    void Reset()
    {
        u8 i = 0;
        for (; i<InputStick_COUNT; i++)
        {
            stickLeft[0] = 0.0f;
            stickLeft[1] = 0.0f;
        }
        for (i = 0; i<InputButton_COUNT; i++)
        {
            buttons[i] = InputButtonState_OFF;
        }
    }
};

class Game;

class Input
{
private:
    InputKeyboardKey keyboardBuffer[KEYBOARD_BUFFER_SIZE];
    u8 keyboardBufferPos = 0;

    u8 joystickCount = 0;
    bool isDiscovering = false;

    InputGamepad gamepads[INPUT_GAMEPAD_COUNT];
    InputKeyboardMap keyboardMap;

    static InputButton JoyToGamepadButton(i8 button);

    Input();

public:
    static Input & Inst();
    
    void UpdatePre();
    void UpdatePost();
    void EventTick();
    void Reset();
    static bool IsHeld(InputButton button);
    static bool WasPressed(InputButton button);
    static Vector2 GetStick(InputStick stick);
    static bool IsKeyHeld(u16 keyOrScancode);
    static bool WasKeyPressed(u16 keyOrScancode);

    void SetDiscovering(bool value) { isDiscovering = value; }
    inline bool IsDiscovering() const { return isDiscovering; }

    void ResetKeyboardMapping();

    friend class Game;

private:
    void HandleEvent(SDL_Event & e);
    void HandleKeyboard(SDL_Event & e);
    void HandleJoystickButton(SDL_Event & e);
    void HandleJoystickMotion(SDL_Event & e);
    InputGamepad * GetGamepadByJoystickId(i32 joystickId);
    InputButtonState GetKeyState(u16 scancode) const;
    InputButtonState GetKeyState(u16 scancodes[2]) const;
    InputButtonState GetKeyState(InputKeyboardBinding binding) const;
};

#endif