#include <Input.h>
#include <Maths.h>
#include <desslog.h>

#define DUMMY_BUFFER_SIZE 16 //__TIME__[6] + __TIME__[7]
int DUMMY_BUFFER[DUMMY_BUFFER_SIZE];

Input & Input::Inst()
{
    static Input first;
    return first;
}

Input::Input()
{
    if (SDL_Init(SDL_INIT_JOYSTICK) < 0) 
    {
		Log_Error("SDL joystick initialization failed.  Error: %s\n", SDL_GetError());
	}

    Reset();
    ResetKeyboardMapping();
}

void Input::ResetKeyboardMapping()
{
    keyboardMap.Unbind();
    keyboardMap.moveLeft[0] = SDL_SCANCODE_LEFT;
    keyboardMap.moveLeft[1] = SDL_SCANCODE_A;
    keyboardMap.moveRight[0] = SDL_SCANCODE_RIGHT;
    keyboardMap.moveRight[1] = SDL_SCANCODE_D;
    keyboardMap.moveUp[0] = SDL_SCANCODE_UP;
    keyboardMap.moveUp[1] = SDL_SCANCODE_W;
    keyboardMap.moveDown[0] = SDL_SCANCODE_DOWN;
    keyboardMap.moveDown[1] = SDL_SCANCODE_S;
    keyboardMap.SetButtonBinding(InputButton_A, SDL_SCANCODE_SPACE);
    keyboardMap.SetButtonBinding(InputButton_B, SDL_SCANCODE_LSHIFT, SDL_SCANCODE_RSHIFT);
    keyboardMap.SetButtonBinding(InputButton_START, SDL_SCANCODE_RETURN);
    keyboardMap.SetButtonBinding(InputButton_BACK, SDL_SCANCODE_ESCAPE);
}

void Input::Reset()
{
    for (u16 i = 0; i < KEYBOARD_BUFFER_SIZE; i++)
    {
        keyboardBuffer[i].key = '\0';
        keyboardBuffer[i].scancode = 0;
        keyboardBuffer[i].state = InputButtonState_OFF;
    }
    for (u16 i = 0; i < INPUT_GAMEPAD_COUNT; i++)
    {
        gamepads[i].Reset();
    }
}

void Input::HandleEvent(SDL_Event & e)
{
    if ((e.type == SDL_KEYDOWN || e.type == SDL_KEYUP) && e.key.keysym.scancode > 0)
    {
        HandleKeyboard(e);
    }
    else if (e.type == SDL_JOYBUTTONDOWN || e.type == SDL_JOYBUTTONUP)
    {
        HandleJoystickButton(e);
    }
    else if (e.type == SDL_JOYAXISMOTION)
    {
        HandleJoystickMotion(e);
    }
}

void Input::HandleKeyboard(SDL_Event & e)
{
    u8 kbi = 0;
    for (; kbi < KEYBOARD_BUFFER_SIZE; kbi++)
    {
        if (keyboardBuffer[kbi].scancode == e.key.keysym.scancode)
            break;
    }
    if (kbi == KEYBOARD_BUFFER_SIZE)
    {
        for (kbi = 0; kbi < KEYBOARD_BUFFER_SIZE; kbi++)
        {
            keyboardBufferPos++;
            keyboardBufferPos = keyboardBufferPos % ((u8)KEYBOARD_BUFFER_SIZE);   
            if (keyboardBuffer[keyboardBufferPos].state < InputButtonState_PRESSED)
                break;
        }
        kbi = keyboardBufferPos;
    }
    if (e.type == SDL_KEYDOWN)
    {
        keyboardBuffer[kbi].scancode = e.key.keysym.scancode;
        keyboardBuffer[kbi].key = e.key.keysym.sym;
        if (keyboardBuffer[kbi].state < InputButtonState_PRESSED)
        {
            keyboardBuffer[kbi].state = InputButtonState_PRESSED;
        }
    }
    else
    {
        if (keyboardBuffer[kbi].scancode == e.key.keysym.scancode)
        {
            keyboardBuffer[kbi].state = InputButtonState_RELEASED;
        }
    }
}

InputButton Input::JoyToGamepadButton(i8 button)
{
    switch (button)
    {
        case SDL_CONTROLLER_BUTTON_A:
        case SDL_CONTROLLER_BUTTON_X:
            return InputButton_A;
        case SDL_CONTROLLER_BUTTON_B:
        case SDL_CONTROLLER_BUTTON_Y:
            return InputButton_B;
        case SDL_CONTROLLER_BUTTON_START:
            return InputButton_START;
        case SDL_CONTROLLER_BUTTON_BACK:
            return InputButton_BACK;
    }
    return InputButton_NONE;
}

void Input::HandleJoystickButton(SDL_Event & e)
{
    InputGamepad * gamepad = GetGamepadByJoystickId(e.jbutton.which);
    if (gamepad == NULL)
        return;
    
    InputButton button = JoyToGamepadButton(e.jbutton.button);
    if (e.jbutton.type == SDL_JOYBUTTONDOWN)
    {
        if (gamepad->buttons[button] < InputButtonState_PRESSED)
        {
            gamepad->buttons[button] = InputButtonState_PRESSED;
        }
    }
    else if (e.jbutton.type = SDL_JOYBUTTONUP)
    {
        gamepad->buttons[button] = InputButtonState_RELEASED;
    }
}

void Input::HandleJoystickMotion(SDL_Event & e)
{
    InputGamepad * gamepad = GetGamepadByJoystickId(e.jbutton.which);
    if (gamepad == NULL)
        return;

    if (e.jaxis.axis > 1)
        return;
    
    InputAxis axis = e.jaxis.axis == 0 ? InputAxis_H : InputAxis_V;
    gamepad->stickLeft[axis] = (float)e.jaxis.value / 32767.0f;
}

void Input::UpdatePre()
{   
    if (isDiscovering)
    {
        int newJoystickCount = SDL_NumJoysticks();
        if (newJoystickCount != joystickCount)
        {
            joystickCount = newJoystickCount;
            Log_Info("%i joysticks connected.\n", joystickCount);
            
            for (u8 i = 0; i < INPUT_GAMEPAD_COUNT; i++)
            {
                gamepads[i].joytickId = -1;
            }

            u8 gamepadI = 0;
            for (u8 i=0; i<joystickCount; i++)
            {
                SDL_Joystick * joystick = SDL_JoystickOpen(i);
                if (joystick == NULL)
                {
                    Log_Error("Bad joystick %s\n", SDL_GetError());
                    continue;
                }
                gamepads[gamepadI].joytickId = SDL_JoystickInstanceID(joystick);
                gamepadI++;
                if (gamepadI == INPUT_GAMEPAD_COUNT)
                    break;
            }
        }
    }

    for (u8 i = 0; i < INPUT_GAMEPAD_COUNT; i++)
    {
        gamepads[i].Clamp();
    }
}

void Input::UpdatePost()
{
    for (u8 i = 0; i < KEYBOARD_BUFFER_SIZE; i++)
    {
        switch (keyboardBuffer[i].state)
        {
            case InputButtonState_PRESSED:
                keyboardBuffer[i].state = InputButtonState_HELD;
                break;

            case InputButtonState_RELEASED:
                keyboardBuffer[i].state = InputButtonState_OFF;
                keyboardBuffer[i].scancode = 0;
                keyboardBuffer[i].key = '\0';
                break;
        }
    }
    
    for (u8 i = 0; i < INPUT_GAMEPAD_COUNT; i++)
    {
        for (u8 j = 0; j < InputButton_COUNT; j++)
        {
            if (gamepads[i].buttons[j] == InputButtonState_PRESSED)
            {
                gamepads[i].buttons[j] = InputButtonState_HELD;
            }
            if (gamepads[i].buttons[j] == InputButtonState_RELEASED)
            {
                gamepads[i].buttons[j] = InputButtonState_OFF;
            }
        }
    }
}

bool Input::IsHeld(InputButton button)
{
    for (size_t i = 0; i < INPUT_GAMEPAD_COUNT; i++)
    {
        if (Inst().gamepads[i].IsHeld(button))
            return true;
    }
    if (Inst().GetKeyState(Inst().keyboardMap.GetButtonBinding(button)) >= InputButtonState_PRESSED)
        return true;
    return false;
}

bool Input::WasPressed(InputButton button)
{
    for (size_t i = 0; i < INPUT_GAMEPAD_COUNT; i++)
    {
        if (Inst().gamepads[i].WasPressed(button))
            return true;
    }
    if (Inst().GetKeyState(Inst().keyboardMap.GetButtonBinding(button)) == InputButtonState_PRESSED)
        return true;
    return false;
}

Vector2 Input::GetStick(InputStick stick)
{
    Vector2 result = Vector2_ZERO;
    for (size_t i = 0; i < INPUT_GAMEPAD_COUNT; i++)
    {
        result += Inst().gamepads[i].GetStick(stick);
    }
    if (stick == InputStick_LEFT && Maths::LengthSq(result) < 0.1f)
    {
        if (Inst().GetKeyState(Inst().keyboardMap.moveLeft) >= InputButtonState_PRESSED)
        {
            result.x = -1.0f;
        }
        else if (Inst().GetKeyState(Inst().keyboardMap.moveRight) >= InputButtonState_PRESSED)
        {
            result.x = 1.0f;
        }
        if (Inst().GetKeyState(Inst().keyboardMap.moveUp) >= InputButtonState_PRESSED)
        {
            result.y = -1.0f;
        }
        else if (Inst().GetKeyState(Inst().keyboardMap.moveDown) >= InputButtonState_PRESSED)
        {
            result.y = 1.0f;
        }
    }
    result = Maths::Clamp(result, 1.0f);
    return result;
}

bool Input::IsKeyHeld(u16 key)
{
    InputKeyboardKey * kb = Inst().keyboardBuffer;
    for (u8 i = 0; i < KEYBOARD_BUFFER_SIZE; i++)
    {
        if (kb[i].state >= InputButtonState_PRESSED && (kb[i].key == key || kb[i].scancode == key))
            return true;
    }
    return false;
}

bool Input::WasKeyPressed(u16 key)
{
    InputKeyboardKey * kb = Inst().keyboardBuffer;
    for (u8 i = 0; i < KEYBOARD_BUFFER_SIZE; i++)
    {
        if (kb[i].state == InputButtonState_PRESSED && (kb[i].key == key || kb[i].scancode == key))
            return true;
    }
    return false;
}

InputButtonState Input::GetKeyState(u16 scancode) const
{
    if (scancode > 0)
    {
        for (u8 i = 0; i < KEYBOARD_BUFFER_SIZE; i++)
        {
            if (keyboardBuffer[i].scancode == scancode)
                return keyboardBuffer[i].state;
        }
    }
    return InputButtonState_OFF;
}

InputButtonState Input::GetKeyState(u16 scancodes[2]) const
{
    InputButtonState stateA = GetKeyState(scancodes[0]);
    InputButtonState stateB = GetKeyState(scancodes[1]);
    return stateA >= stateB ? stateA : stateB;
}

InputButtonState Input::GetKeyState(InputKeyboardBinding binding) const
{
    InputButtonState stateA = GetKeyState(binding.key);
    InputButtonState stateB = GetKeyState(binding.alt);
    return stateA >= stateB ? stateA : stateB;
}

InputGamepad * Input::GetGamepadByJoystickId(i32 joystickId)
{
    for (u8 i=0; i<INPUT_GAMEPAD_COUNT; i++)
    {
        if (gamepads[i].joytickId == joystickId)
            return &gamepads[i];
    }
    return NULL;
}